<?php

namespace Mtc\Returns\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Mtc\Orders\Contracts\OrderContract;
use Mtc\Returns\Http\Requests\ValidReturnUpdateRequest;
use Mtc\Returns\ReturnReason;
use Mtc\Returns\ReturnRequest;
use Mtc\Shop\Order\Item;

/**
 * Class AdminReturnController
 *
 * @package Mtc\Returns
 */
class AdminReturnController extends Controller
{
    /**
     * Render the page
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index(Request $request)
    {
        $this->page_meta['title'] = 'Return Requests';
        return template('admin/returns/index.twig', [
            'page_meta' => $this->page_meta,
            'returns' => ReturnRequest::query()
                ->with([
                    'items',
                    'order'
                ])
                ->search($request)
                ->latest()
                ->paginate(50),
        ]);
    }

    /**
     * Create return request for an order
     *
     * @param Request $request
     * @param ReturnRequest $return
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request, ReturnRequest $return, OrderContract $order)
    {
        $order = $order->newQuery()->find($request->input('order_id'));
        $return->order_id = $request->input('order_id');
        $return->reference = $order->reference;
        $return->status_id = 1;
        $return->save();

        Item::query()
            ->where('order_id', $request->input('order_id'))
            ->get()
            ->each(function ($order_item) use ($return) {
                $return->items()
                    ->create([
                        'order_item_id' => $order_item->id,
                        'purchasable_id' => $order_item->purchasable_id,
                        'purchasable_type' => $order_item->purchasable_type,
                        'requested_quantity' => 0,
                        'reason_id' => 0,
                    ]);
            });


        $return->history()
            ->create([
                'user_id' => Auth::user()->id,
                'text' => 'Created return request',
            ]);

        session()->flash('success', 'Return updated');
        return redirect()->to(route('admin.returns.edit', [ $return->id ]));
    }

    /**
     * Render view for processing a return
     *
     * @param ReturnRequest $return
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function edit(ReturnRequest $return)
    {
        $this->page_meta['title'] = 'Manage Return Request';
        return template('admin/returns/edit.twig', [
            'page_meta' => $this->page_meta,
            'return' => $return,
            'reasons' => ReturnReason::listAll(),
            'statuses' => ReturnRequest::statusNames()
        ]);
    }

    /**
     *
     * @param ValidReturnUpdateRequest $request
     * @param ReturnRequest $return
     */
    public function update(ValidReturnUpdateRequest $request, ReturnRequest $return)
    {
        $change_list = collect([]);
        $return->items
            ->each(function ($item) use ($request, $change_list) {
                $item->requested_quantity = $request->input('requested_quantity.' . $item->id, 0);
                $item->approved_quantity = $request->input('approved_quantity.' . $item->id, 0);
                if (config('returns.refund_amounts_enabled') === true) {
                    $item->refunded_amount = $request->input('refunded_amount.' . $item->id, 0);
                }
                $item->reason_id = $request->input('reason_id.' . $item->id, 0);
                foreach ($item->getDirty() as $field => $value) {
                    if ($field === 'reason_id') {
                        $value = ReturnReason::query()->find($value)->name;
                        $field = 'Reason';
                    } else {
                        $field = ucfirst(str_replace('_', ' ', $field));
                    }
                    $change_list->push($item->orderItem->name . " - {$field} changed to {$value}");
                }
                $item->save();
            });

        if ($request->has('delivery_refund_amount')) {
            $return->delivery_refund_amount = $request->input('delivery_refund_amount', 0);
            foreach ($return->getDirty() as $field => $value) {
                $field = ucfirst(str_replace('_', ' ', $field));
                $change_list->push("{$field} changed to {$value}");
            }
            $return->save();
        }

        if ($change_list->isNotEmpty()) {
            $return->history()
                ->create([
                    'user_id' => Auth::user()->id,
                    'text' => $change_list->implode(PHP_EOL),
                ]);
        }
        session()->flash('success', 'Return updated');
        return redirect()->back();
    }

    /**
     *
     * @param ValidReturnUpdateRequest $request
     * @param ReturnRequest $return
     */
    public function updateStatus(Request $request, ReturnRequest $return_request)
    {
        $return_request->status_id = $request->input('status_id');
        if ($return_request->isDirty('status_id') == false) {
            return redirect()->back();
        }

        if (
            $return_request->status_id == config('returns.statuses.processed')
            && config('refunds.enabled') && $return_request->hasRefundItems()
        ) {
            return redirect()->to(route('refunds.handle-request', [ $return_request->id ]));
        }

        $statuses = ReturnRequest::statusNames();
        $return_request->save();

        $return_request->history()
            ->create([
                'user_id' => Auth::user()->id,
                'text' => 'Changed status to ' . $statuses[$return_request->status_id],
            ]);

        session()->flash('success', 'Return updated');
        return redirect()->back();
    }

    /**
     * Cancel a return
     *
     * @param ReturnRequest $return
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(ReturnRequest $return)
    {
        $return->cancel();
        session()->flash('success', 'Return Cancelled');
        return redirect()->back();
    }
}
