<?php

namespace Mtc\Returns\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Mtc\Orders\Contracts\OrderContract;
use Mtc\Returns\Http\Requests\ValidReturnUpdateRequest;
use Mtc\Returns\ReturnReason;
use Mtc\Returns\ReturnRequest;
use Mtc\Shop\Order\Item;

/**
 * Class AdminReturnController
 *
 * @package Mtc\Returns
 */
class AdminReturnController extends Controller
{
    /**
     * Render the page
     *
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function index()
    {
        $this->page_meta['title'] = 'Return Requests';
        return template('admin/returns/index.twig', [
            'page_meta' => $this->page_meta,
            'returns' => ReturnRequest::query()
                ->with([
                    'items',
                    'order'
                ])
                ->latest()
                ->paginate(50),
        ]);
    }

    /**
     * Create return request for an order
     *
     * @param Request $request
     * @param ReturnRequest $return
     * @return \Illuminate\Http\RedirectResponse
     */
    public function store(Request $request, ReturnRequest $return, OrderContract $order)
    {
        $order = $order->newQuery()->find($request->input('order_id'));
        $return->order_id = $request->input('order_id');
        $return->reference = $order->reference;
        $return->status_id = 1;
        $return->save();

        Item::query()
            ->where('order_id', $request->input('order_id'))
            ->get()
            ->each(function ($order_item) use ($return) {
                $return->items()
                    ->create([
                        'order_item_id' => $order_item->id,
                        'purchasable_id' => $order_item->purchasable_id,
                        'purchasable_type' => $order_item->purchasable_type,
                        'requested_quantity' => 0,
                        'reason_id' => 0,
                    ]);
            });


        $return->history()
            ->create([
                'user_id' => Auth::user()->id,
                'text' => 'Created return request',
            ]);

        session()->flash('success', 'Return updated');
        return redirect()->to(route('admin.returns.edit', [ $return->id ]));

    }

    /**
     * Render view for processing a return
     *
     * @param ReturnRequest $return
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function edit(ReturnRequest $return)
    {
        $this->page_meta['title'] = 'Manage Return Request';
        return template('admin/returns/edit.twig', [
            'page_meta' => $this->page_meta,
            'return' => $return,
            'reasons' => ReturnReason::listAll(),
            'statuses' => ReturnRequest::statusNames()
        ]);

    }

    /**
     *
     * @param ValidReturnUpdateRequest $request
     * @param ReturnRequest $return
     */
    public function update(ValidReturnUpdateRequest $request, ReturnRequest $return)
    {
        $change_list = collect([]);
        $return->items
            ->each(function ($item) use ($request, $change_list) {
                $item->requested_quantity = $request->input('requested_quantity.' . $item->id, 0);
                $item->approved_quantity = $request->input('approved_quantity.' . $item->id, 0);
                $item->reason_id = $request->input('reason_id.' . $item->id, 0);
                foreach ($item->getDirty() as $field => $value) {
                    if ($field === 'reason_id') {
                        $value = ReturnReason::query()->find($value)->name;
                        $field = 'Reason';
                    } else {
                        $field = ucfirst(str_replace('_', ' ', $field));
                    }
                    $change_list->push($item->orderItem->name . " - {$field} changed to {$value}");
                }
                $item->save();
            });

        $return->status_id = $request->input('status_id');
        if ($return->isDirty('status_id')) {
            $statuses = ReturnRequest::statusNames();
            $change_list->push('Changed status to ' . $statuses[$return->status_id]);
        }

        $return->save();

        if (!empty($change_list)) {
            $return->history()
                ->create([
                    'user_id' => Auth::user()->id,
                    'text' => $change_list->implode(PHP_EOL),
                ]);
        }
        session()->flash('success', 'Return updated');
        return redirect()->back();
    }

    /**
     * Cancel a return
     *
     * @param ReturnRequest $return
     * @return \Illuminate\Http\RedirectResponse
     */
    public function destroy(ReturnRequest $return)
    {
        $return->cancel();
        session()->flash('success', 'Return Cancelled');
        return redirect()->back();
    }
}
