<?php

namespace Mtc\Reports;

use Carbon\Carbon;
use Exception;
use Illuminate\Contracts\Pagination\LengthAwarePaginator;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Mail;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Maatwebsite\Excel\Facades\Excel;
use Mtc\Reports\Contracts\ReportLogModel;
use Mtc\Reports\Contracts\ReportModel;
use Mtc\Reports\Exports\ReportExport;
use Mtc\Reports\Http\Requests\UpdateReportRequest;

use Mtc\Reports\Mail\ReportMail;

use function PHPUnit\Framework\matches;

class ReportRepository
{
    protected Collection $data;
    protected Builder $query;

    /**
     * @var ReportModel|Model
     */
    protected $report;

    protected $attachment = null;

    public function __construct(protected ReportModel $model, protected ReportLogModel $logModel)
    {
        //
    }

    /**
     * Get list of reports
     *
     * @param Request $request
     * @return LengthAwarePaginator
     */
    public function getList(Request $request): LengthAwarePaginator
    {
        return $this->model->newQuery()
            ->active($request->input('active'))
            ->paginate();
    }

    /**
     * View details of a report
     *
     * @param ReportModel $report
     * @return ReportModel
     */
    public function show(ReportModel $report): ReportModel
    {
        $report->load('logs');
        return $report;
    }

    /**
     * Create a report
     *
     * @param string $name
     * @param string $type
     * @return Model|ReportModel
     */
    public function create(string $name, string $type): ReportModel
    {
        $this->report = $this->model->newQuery()
            ->create([
                'name' => $name,
                'type' => $type,
            ]);

        return $this->report;
    }

    /**
     * Update a report
     *
     * @param UpdateReportRequest $request
     * @param ReportModel $report
     * @return ReportModel
     */
    public function update(UpdateReportRequest $request, ReportModel $report)
    {
        $this->report = $report;

        $this->report->fill([
            'name' => $request->input('name'),
            'type' => $request->input('type'),
            'active' => $request->input('active'),
            'schedule' => $request->input('schedule'),
            'conditions' => $request->input('conditions'),
            'columns' => $request->input('columns'),
            'recipient' => $request->input('recipient'),
            'format' => $request->input('format'),
        ]);

        $this->setSchedule();
        $this->report->save();

        return $this->report;
    }

    /**
     * Get due reports
     *
     * @return Collection
     */
    public function getDueReports(): Collection
    {
        return $this->model->newQuery()
            ->active(1)
            ->where('next_report_due_at', '<=', Carbon::now())
            ->get();
    }

    /**
     * Run a report
     * Get data, send report and update next report time
     *
     * @param ReportModel $report
     * @return void
     */
    public function run(ReportModel $report)
    {
        try {
            $this->report = $report;
            $success = $this->buildQuery()
                ->getData()
                ->notify();

            $this->report->logs()
                ->create([
                    'succeeded' => $success,
                ]);

            $this->report->update([
                'next_report_due_at' => $this->report->getNextDueAt(Carbon::now()),
                'last_report_sent_at' => Carbon::now(),
            ]);
            $this->cleanOldLogs();
        } catch (Exception $exception) {
            $this->report->logs()
                ->create([
                    'succeeded' => false,
                    'details' => $exception->getMessage(),
                ]);
        }
    }

    public function getReport()
    {
        return $this->report;
    }

    /**
     * Delete a report
     *
     * @param int $reportId
     * @return bool
     */
    public function destroy(int $reportId): bool
    {
        try {
            $this->model->newQuery()
                ->where('id', $reportId)
                ->delete();

            $this->logModel->newQuery()
                ->where('report_id', $reportId)
                ->delete();

            return true;
        } catch (\Exception $exception) {
            Log::error($exception->getMessage(), [
                'report_id' => $reportId,
            ]);
            return false;
        }
    }

    /**
     * Start eloquent query
     *
     * @return $this
     */
    protected function buildQuery()
    {
        $this->query = App::make(Config::get('reports.types.' . $this->report->type))->query();
        $this->query->select($this->report->columns);
        $this->applyQueryConditions($this->query, $this->report->conditions);

        return $this;
    }

    /**
     * Apply query conditions (nested)
     *
     * @param Builder $query
     * @param $conditions
     * @return void
     */
    protected function applyQueryConditions(Builder $query, $conditions)
    {
        foreach ($conditions as $condition) {
            match ($condition['type']) {
                'and' => $query->where($condition['column'], $condition['operator'], $condition['value']),
                'or' => $query->orWhere($condition['column'], $condition['operator'], $condition['value']),
                'has' => $query->whereHas($condition['column']),
                'doesnt-have' => $query->whereDoesntHave($condition['column']),
                'null' => $query->whereNull($condition['column']),
                'or-null' => $query->orWhereNull($condition['column']),
                'not-null' => $query->whereNotNull($condition['column']),
                'or-not-null' => $query->orWhereNotNull($condition['column']),
                'where-sub' => $query->where(
                    fn($query) => $this->applyQueryConditions($query, $conditions['conditions'])
                ),
                'or-where-sub' => $query->orWhere(
                    fn($query) => $this->applyQueryConditions($query, $conditions['conditions'])
                ),
            };
        }
    }

    /**
     * Get report data
     *
     * @return $this
     * @throws \Illuminate\Contracts\Filesystem\FileNotFoundException
     */
    protected function getData()
    {
        $this->data = $this->query->get();
        if ($this->report->sendAsFile()) {
            $this->generateAttachment();
        }
        return $this;
    }

    /**
     * Sent report via email
     *
     * @return bool
     */
    protected function notify(): bool
    {
        $recipients = collect(explode(',', $this->report->recipient))
            ->map(fn($recipient) => trim($recipient))
            ->filter(fn($recipient) => $recipient && filter_var($recipient, FILTER_VALIDATE_EMAIL))
            ->each(fn($recipient) => Mail::to($recipient)
                ->send(new ReportMail($this->data, $this->report, $this->attachment)));
        return true;
    }

    /**
     * Generate attachment for export
     *
     * @return void
     * @throws \Illuminate\Contracts\Filesystem\FileNotFoundException
     */
    protected function generateAttachment()
    {
        $fileName = Str::slug($this->report->name . Carbon::now()->format('Y-m-d-H-i')) . '.' . $this->report->format;
        $exportClass = $this->report->exportClass();
        Excel::store(
            new $exportClass($this->data),
            $fileName,
            config('reports.report-storage-disk'),
            null,
            ['visibility' => 'private']
        );

        $this->attachment = Attachment::fromData(
            fn() => Storage::disk(config('reports.report-storage-disk'))->get($fileName),
            $fileName
        );
    }

    /**
     * Perform self-cleanup of report logs
     *
     * @return void
     */
    protected function cleanOldLogs()
    {
        $length = Config::get('reports.log_length');
        if ($length === null) {
            return;
        }

        $this->report->logs()
            ->whereNotIn('id', $this->report->logs()->latest()->limit($length)->pluck('id'))
            ->delete();
    }

    /**
     * Set the schedule for next report
     *
     * @return void
     */
    protected function setSchedule()
    {
        if ($this->report->active !== true) {
            $this->report->next_report_due_at = null;
        }

        if ($this->report->isDirty('schedule')) {
            $this->report->next_report_due_at = $this->report->getNextDueAt(Carbon::now());
        }
    }
}
