<?php

namespace Mtc\Refunds\Http\Controllers;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Mtc\Checkout\Invoice\Payment;
use Mtc\Refunds\Facades\Refunds;
use Mtc\Refunds\Refund;
use Mtc\Returns\ReturnRequest;
use Mtc\Orders\Item;

/**
 * Class ReturnController
 *
 * @package Mtc\Refunds
 */
class RefundController extends Controller
{
    /**
     * Handle refund request for payment
     *
     * @param Request $request
     * @param Payment $payment
     * @return \Illuminate\Http\RedirectResponse
     */
    public function payment(Request $request, Payment $payment)
    {
        $result = Refunds::refundPayment($payment, $request->input('amount'));
        if (empty($result)) {
            session()->flash('error', __('refunds::refunds.failed_refunding_payment'));
            return back();
        }

        Refund::query()
            ->create([
                'invoice_id' => $payment->invoice_id,
                'payment_id' => $payment->id,
                'amount' => $result['amount'],
                'reference' => $result['reference'],
            ]);

        session()->flash('success', __('refunds::refunds.payment_refund_successful'));
        return back();
    }

    /**
     * Ask to confirm amount on return request
     *
     * @param ReturnRequest $return_request
     * @return \Illuminate\Contracts\View\Factory|\Illuminate\View\View
     */
    public function viewReturnRequest(ReturnRequest $return_request)
    {
        $this->page_meta['title'] = __('refunds::refunds.confirm_amount_title');

        $refund_amount = $return_request->items
            ->filter(function ($item) {
                return $item->approved_quantity > 0;
            })
            ->sum(function ($item) {
                return $item->orderItem->paid_price->raw() * $item->approved_quantity;
            });

        $payments = Payment::query()
            ->whereHas('invoice', function ($invoice_query) use ($return_request) {
                return $invoice_query->whereHas('items', function ($invoice_item_query) use ($return_request) {
                    return $invoice_item_query->where('parent_type', Item::class)
                        ->whereIn('parent_id', $return_request->items->pluck('order_item_id'));
                });
            })
            ->get();

        return template('refunds/confirm_amount.twig', [
            'return_request' => $return_request,
            'page_meta' => $this->page_meta,
            'refund_amount' => $refund_amount,
            'payments' => $payments,
        ]);
    }

    /**
     * Refund a return request
     *
     * @param Request $request
     * @param ReturnRequest $return_request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function returnRequest(Request $request, ReturnRequest $return_request)
    {
        $refunded_payments = collect($request->input('amount'))
            ->filter()
            ->filter(function ($amount, $payment_id) use ($return_request) {
                $payment = Payment::query()->find($payment_id);
                $result = Refunds::refundPayment($payment, $amount);
                if (empty($result)) {
                    return false;
                }

                Refund::query()
                    ->create([
                        'invoice_id' => $payment->invoice_id,
                        'payment_id' => $payment->id,
                        'return_request_id' => $return_request->id,
                        'amount' => $result['amount'],
                        'reference' => $result['reference'],
                    ]);
                return true;
            });

        if ($refunded_payments->isEmpty()) {
            session()->flash('error', __('refunds::refunds.failed_refunding_payment'));
            return redirect()->to(route('admin.returns.edit', [ $return_request->id ]));
        }

        $statuses = ReturnRequest::statusNames();
        $return_request->status_id = config('returns.statuses.processed');
        $return_request->save();

        $return_request->history()
            ->create([
                'user_id' => Auth::user()->id,
                'text' => 'Changed status to ' . $statuses[$return_request->status_id],
            ]);

        session()->flash('success', __('refunds::refunds.payment_refund_successful'));
        return redirect()->to(route('admin.returns.edit', [ $return_request->id ]));
    }
}
