<?php

namespace Mtc\Realex;

use Exception;
use GlobalPayments\Api\Entities\Address;
use GlobalPayments\Api\Entities\Enums\AddressType;
use GlobalPayments\Api\Entities\Enums\HppVersion;
use GlobalPayments\Api\Entities\HostedPaymentData;
use GlobalPayments\Api\HostedPaymentConfig;
use GlobalPayments\Api\Services\HostedService;
use GlobalPayments\Api\ServicesConfig;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\PayableContract;
use Mtc\Checkout\Contracts\PaymentGateway;
use Mtc\Checkout\Invoice\Payment;
use Mtc\Checkout\PaymentForm;
use Mtc\Members\Member;

/**
 * Realex Payment Gateway
 *
 * @package  Mtc\Realex
 * @author   Martins Fridenbergs <martins.fridenbergs@mtcmedia.co.uk>
 */
class Realex implements PaymentGateway
{
    /**
     * Check if the gateway is available for use on this payment.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return bool
     */
    public function isApplicable(InvoiceRepositoryContract $invoice, $payable): bool
    {
        if ($invoice->getOutstandingAmount() <= 0.01) {
            return false;
        }

        return App::make(config('realex.applicable_check_class'))->handle($invoice, $payable);
    }

    /**
     * Render the payment template.
     *
     * @param InvoiceRepositoryContract $invoice
     * @return PaymentForm
     */
    public function getPaymentForm(InvoiceRepositoryContract $invoice): PaymentForm
    {
        $member = Members::member();
        if ($member && config('realex.use_card_storage')) {
            $customer_details = (new Customer)->findCustomer($member);
        } else {
            $customer_details = null;
        }

        $service = new HostedService($this->loadConfig($member, $customer_details, $invoice->getCurrency()));
        $payment_data = new HostedPaymentData();

        if ($member && config('realex.use_card_storage')) {
            $payment_data->offerToSaveCard = true;
            $payment_data->customerExists = !empty($customer_details['customer_token']);
            if (!empty($customer_details['customer_token'])) {
                $payment_data->customerKey = $customer_details['customer_token'];
                $payment_data->paymentKey = $customer_details['payment_token'];
            }
        }

        $order = $invoice->getPayable();
        $payment_data->customerEmail = $order->email;
        $payment_data->customerPhoneMobile = str_replace('+44', '44|', $order->customer_number);
        $payment_data->addressesMatch = false;

        $order_billing_address = $invoice->getPayable()->billingAddress;
        $order_shipping_address = $invoice->getPayable()->shippingAddress;

        $shippingAddress = new Address();
        $shippingAddress->streetAddress1 = $order_shipping_address->address1;
        $shippingAddress->streetAddress2 = $order_shipping_address->address2;
        $shippingAddress->city = $order_shipping_address->city;
        $shippingAddress->state = $order_shipping_address->state;
        $shippingAddress->postalCode = $order_shipping_address->postcode;
//        $shippingAddress->country = $order_shipping_address->country;
        // @todo make dynamic
        $shippingAddress->country = 826;

        $billingAddress = new Address();
        $billingAddress->streetAddress1 = $order_billing_address->address1;
        $billingAddress->streetAddress2 = $order_billing_address->address2;
        $billingAddress->city = $order_billing_address->city;
        $billingAddress->state = $order_billing_address->state;
        $billingAddress->postalCode = $order_billing_address->postcode;
//        $billingAddress->country = $order_billing_address->country;
        // @todo make dynamic
        $billingAddress->country = 826;

        if (config('realex.multi_currency_enabled')) {
            $json = $service->charge($invoice->getOutstandingAmountInCurrency())
                ->withCurrency($invoice->getCurrency())
                ->withHostedPaymentData($payment_data)
                ->withAddress($billingAddress, AddressType::BILLING)
                ->withAddress($shippingAddress, AddressType::SHIPPING)
                ->serialize();
        } else {
            $json = $service->charge($invoice->getOutstandingAmount())
                ->withCurrency(config('currencies.default_currency'))
                ->withHostedPaymentData($payment_data)
                ->withAddress($billingAddress, AddressType::BILLING)
                ->withAddress($shippingAddress, AddressType::SHIPPING)
                ->serialize();
        }

        return new PaymentForm('realex-payment', 'vue-component', [
            'gateway_url' => $this->getApiEndpoint(),
            'pay_url' => route('charge_payment', [ $invoice->getId() ]) . '?gateway=realex',
            'merchant_id' => config('realex.merchant_id'),
            'account_id' => config('realex.account_id'),
            'show_test_cards' => !config('realex.use_production_endpoint', false),
            'json' => $json,
            'name' => __('realex::realex.payment_option_name')
        ]);
    }

    /**
     * Charge payment on invoice
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return bool
     * @throws Exception
     */
    public function charge(Request $request, InvoiceRepositoryContract $invoice): array
    {
        $member = $invoice->getMember();
        if ($member && config('realex.use_card_storage')) {
            $customer_details = (new Customer)->findCustomer($member);
        } else {
            $customer_details = null;
        }

        // payment details to store or return
        $payment_details = [
            'provider' => 'realex',
            'invoice_id' => $invoice->getId(),
            'amount' => $invoice->getOutstandingAmount(),
            'currency_code' => $invoice->getCurrency(),
        ];

        $service = new HostedService($this->loadConfig($member, $customer_details, $invoice->getCurrency()));

        try {
            // create the response object from the response
            $parsed_response = $service->parseResponse($request->input('hpp_response'), true);
        } catch (\Throwable $exception) {
            // the response could not be decoded/verified
            $payment_details += [
                'failed_at' => now(),
                'failure_details' => $exception->getMessage(),
            ];
            // create a failure record
            Payment::query()->forceCreate($payment_details);

            throw $exception;
        }

        // add the details from the parsed response
        $payment_details += [
            'amount_in_currency' => $parsed_response->responseValues['AMOUNT'] / 100,
            'reference' => $parsed_response->orderId,
            'details' => $parsed_response->responseValues,
        ];

        if ($parsed_response->responseCode !== '00') {
            $payment_details += [
                'failed_at' => now(),
                'failure_details' => $parsed_response->responseMessage,
            ];
            // create a failure record
            Payment::query()->forceCreate($payment_details);

            throw new Exception($parsed_response->responseMessage);
        }

        if ($member) {
            (new Customer)->storeCustomerDetails($member, $parsed_response);
        }

        return $payment_details + [
            'confirmation_status' => config('realex.confirmation_status', null),
            'confirmed_at' => now(),
        ];
    }

    /**
     * Get the payment endpoint for the gateway
     *
     * @return string
     */
    public function getApiEndpoint(): string
    {
        return config('realex.use_production_endpoint', false)
            ? 'https://pay.realexpayments.com/pay'
            : 'https://pay.sandbox.realexpayments.com/pay';
    }

    /**
     * Initialize config for payment
     *
     * @param Member $member
     * @param array $member_details
     * @param string $currency
     * @return ServicesConfig
     */
    protected function loadConfig($member, $member_details, $currency): ServicesConfig
    {
        $config = new ServicesConfig();
        $config->merchantId = config('realex.merchant_id');
        $config->accountId = $this->getAccountId($currency);
        $config->sharedSecret = config('realex.secret');
        $config->serviceUrl = $this->getApiEndpoint();
        $config->hostedPaymentConfig = new HostedPaymentConfig();
        $config->hostedPaymentConfig->version = HppVersion::VERSION_2;
        if ($member && config('realex.use_card_storage')) {
            if (!empty($member_details['customer_token'])) {
                $config->hostedPaymentConfig->displaySavedCards = true;
            } else {
                $config->hostedPaymentConfig->cardStorageEnabled = '1';
            }
        }
        return $config;
    }

    /**
     * Find the relevant account ID based on selected currency
     *
     * @param $currency
     * @return \Illuminate\Config\Repository|mixed
     */
    protected function getAccountId($currency)
    {
        if (config('realex.multi_currency_enabled')) {
            return config("realex.currency_specific_account_ids.{$currency}") ?: config('realex.account_id');
        }

        return config('realex.account_id');
    }
}
