<?php

namespace Mtc\Realex;

use Mtc\Members\Member;

/**
 * Class Customer
 *
 * Realex functionality for Customer / Token handling
 *
 * @package Mtc\Realex
 */
class Customer
{
    /**
     * Find the users stripe ID or create one
     *
     * @param Member $member
     * @return array
     */
    public function findCustomer($member)
    {
        return [
            'customer_token' => $this->getCustomersToken($member, 'realex-customer-token'),
            'payment_token' =>  $this->getCustomersToken($member, 'realex-payment-token'),
        ];
    }

    /**
     * Retrieve customer cards from Realex gateway
     * This method checks if user is logged in and has a Realex customer id
     * If user has customer ID we can check if there are cards bound with this user
     *
     * @param Member $member1
     * @param Object $response
     * @return void
     */
    public function storeCustomerDetails($member, $response)
    {
        if ($this->unableToSaveCustomerDetails($response)) {
            return;
        }

        $this->setCustomersToken($member, $response->responseValues['SAVED_PAYER_REF'], 'realex-customer-token');
        $this->setCustomersToken($member, $response->responseValues['SAVED_PMT_REF'], 'realex-payment-token');
    }

    /**
     * Get the customers Realex ID from saved token information against user
     *
     * @param Member $member
     * @param string $token
     * @param string $token_type
     * @return null
     */
    public function setCustomersToken($member, $token, $token_type)
    {
        return $member->authenticators()
            ->updateOrCreate([
                'provider' => $token_type
            ], [
                'provider' => $token_type,
                'key' => $token,
            ]);
    }

    /**
     * Get the customers Realex ID from saved token information against user
     *
     * @param Member $member
     * @param string $token_type
     * @return null
     */
    public function getCustomersToken($member, $token_type)
    {
        return $member->authenticators()
            ->where('provider', $token_type)
            ->first()
            ->key ?? false;
    }

    /**
     * Check if we can save users payment information
     *
     * @param $response
     * @return bool
     */
    protected function unableToSaveCustomerDetails($response)
    {
        return $response->responseValues['PAYER_SETUP'] !== '00'
            || $response->responseValues['PMT_SETUP'] !== '00';
    }
}
