<?php

namespace Mtc\PinnedItems;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Mtc\PinnedItems\Models\PinnedItemModel;

class PinnedItemRepository
{
    public function __construct(protected array $config)
    {
        //
    }

    /**
     * Add $model to pinned items for $owner
     *
     * @param Model $owner
     * @param Model $pinned_object
     * @param string $category
     */
    public function add(Model $owner, Model $pinned_object, string $category = null): void
    {
        PinnedItemModel::query()
            ->updateOrCreate([
                'owner_id' => $owner->id,
                'owner_type' => $owner->getMorphClass(),
                'category' => $category,
                'pinned_item_id' => $pinned_object->id,
                'pinned_item_type' => $pinned_object->getMorphClass(),
            ]);
    }

    /**
     * Add url to pinned items for $owner
     *
     * @param Model $owner
     * @param string $url
     * @param string $category
     */
    public function addUrl(
        Model $owner,
        string $url,
        string $category = null,
        string $title = '',
        string $subtitle = ''
    ): void {
        PinnedItemModel::query()
            ->updateOrCreate([
                'owner_id' => $owner->id,
                'owner_type' => $owner->getMorphClass(),
                'category' => $category,
                'pinned_url' => $url,
                'title' => $title,
                'subtitle' => $subtitle,
            ]);
    }

    /**
     * Remove $model from pinned items for $owner
     *
     * @param Model $owner
     * @param Model $pinned_object
     * @param string $category
     * @return bool
     */
    public function remove(Model $owner, Model $pinned_object, string $category = null): bool
    {
        $pin = PinnedItemModel::query()
            ->where([
                'owner_id' => $owner->id,
                'owner_type' => $owner->getMorphClass(),
                'category' => $category,
                'pinned_item_id' => $pinned_object->id,
                'pinned_item_type' => $pinned_object->getMorphClass(),
            ])
            ->first();

        return $pin ? $pin->delete() : false;
    }

    /**
     * Remove url from pinned items for $owner
     *
     * @param Model $owner
     * @param string $url
     * @param string $category
     * @return bool
     */
    public function removeUrl(Model $owner, string $url, string $category = null): bool
    {
        $pin = PinnedItemModel::query()
            ->where([
                'owner_id' => $owner->id,
                'owner_type' => $owner->getMorphClass(),
                'category' => $category,
                'pinned_url' => $url,
            ])
            ->first();

            return $pin ? $pin->delete() : false;
    }

    /**
     * Get pinned items by $category for $owner
     *
     * @param Model $owner
     * @param string|null $category
     * @param int|null $count
     * @return Collection
     */
    public function get(Model $owner, string $category = null, int $count = null): Collection
    {
        return PinnedItemModel::query()
            ->with('pinnedItem')
            ->where('owner_id', $owner->id)
            ->where('owner_type', $owner->getMorphClass())
            ->when($category, fn($query) => $query->where('category', $category))
            ->where(fn ($query) => $query->whereHas('pinnedItem')->orWhereNotNull('pinned_url'))
            ->when($count, fn($query) => $query->take($count))
            ->latest('updated_at')
            ->get()
            ->map(fn (PinnedItemModel $view) => $view->pinned_item ?? $view);
    }
}
