<?php

namespace Mtc\MediaManager\Http\Controllers;

use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;
use Illuminate\Support\Facades\Validator;
use Intervention\Image\ImageManager;
use Intervention\Image\Drivers\Imagick\Driver as ImagickDriver;

class MediaManagerController extends Controller
{
    protected $disk;
    protected $config;

    public function __construct()
    {
        $this->disk = Storage::disk(config('mediamanager.disk'));
        $this->config = config('mediamanager');
    }

    /**
     * Display the media manager interface
     */
    public function index()
    {
        $title = 'Media Manager';
        $page_meta['title'] = config('app.name') . ' - ' . $title;

        return template('media-manager/index.twig', [
            'page_meta' => $page_meta,
            'title' => $title,
            'config' => [
                'routes' => [
                    'browse' => route('mediamanager.browse'),
                    'upload' => route('mediamanager.upload'),
                    'delete' => route('mediamanager.delete'),
                    'createFolder' => route('mediamanager.create-folder'),
                    'download' => route('mediamanager.download'),
                ],
                'csrfToken' => csrf_token(),
                'maxFileSize' => $this->config['max_file_size'] ?? (10 * 1024 * 1024),
                'maxUploadFiles' => $this->config['max_upload_files'] ?? 10,
                'allowedExtensions' => $this->config['allowed_extensions'] ?? [
                        'jpg',
                        'jpeg',
                        'png',
                        'gif',
                        'pdf',
                        'docx',
                        'xlsx'
                    ],
                'app_name' => config('app.name')
            ],
        ]);
    }

    /**
     * Browse files and folders
     */
    public function browse(Request $request)
    {
        $path = $request->get('path', '/');
        $path = $this->sanitizePath($path);

        $files = $this->disk->files($path);
        $directories = $this->disk->directories($path);
        $items = [];

        foreach ($directories as $directory) {
            $items[] = [
                'type' => 'dir',
                'name' => basename($directory),
                'path' => $directory,
                'size' => 0,
                'modified' => $this->disk->lastModified($directory)
            ];
        }

        foreach ($files as $file) {
            $mime = $this->disk->mimeType($file);
            if (in_array($mime, $this->config['allowed_mimes'])) {
                $items[] = [
                    'type' => 'file',
                    'name' => basename($file),
                    'path' => $file,
                    'size' => $this->disk->size($file),
                    'modified' => $this->disk->lastModified($file),
                    'mime' => $mime,
                    'url' => $this->disk->url($file)
                ];
            }
        }

        return response()->json([
            'items' => $items,
            'current_path' => $path
        ]);
    }

    /**
     * Handle file uploads
     */
    public function upload(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'files.*' => [
                'required',
                'file',
                'max:' . ($this->config['max_file_size'] / 1024),
                'mimes:' . implode(',', $this->config['allowed_extensions'])
            ],
            'path' => 'required|string'
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $path = $this->sanitizePath($request->input('path'));
        $uploadedFiles = [];

        // Instantiate the ImageManager (you can reuse it inside the loop or outside)
        $imageManager = new ImageManager(new ImagickDriver());

        try {
            foreach ($request->file('files') as $file) {
                $filename = $this->sanitizeFilename($file->getClientOriginalName());
                $filepath = $path . '/' . $filename;

                if (Str::startsWith($file->getMimeType(), 'image/')) {
                    try {
                        // Intervention 3.x syntax
                        $image = $imageManager->read($file);

                        // Resize if too large
                        if (
                            $image->width() > $this->config['image_max_width'] ||
                            $image->height() > $this->config['image_max_height']
                        ) {
                            $image = $image->scaleDown(
                                $this->config['image_max_width'],
                                $this->config['image_max_height']
                            );
                        }

                        // Determine output format
                        $originalExt = strtolower($file->getClientOriginalExtension());
                        $shouldConvertToWebp = $this->config['convert_to_webp'] === true;
                        $targetExt = $shouldConvertToWebp ? 'webp' : $originalExt;

                        // Set final path with extension
                        $finalPath = preg_replace('/\.[^.]+$/', '.' . $targetExt, $filepath);

                        // Encode image based on type
                        switch ($targetExt) {
                            case 'jpg':
                            case 'jpeg':
                                $encoded = $image->toJpeg($this->config['jpeg_quality']);
                                break;
                            case 'png':
                                $encoded = $image->toPng();
                                break;
                            case 'webp':
                                $encoded = $image->toWebp($this->config['jpeg_quality']);
                                break;
                            default:
                                $encoded = $image->toOriginal(); // fallback
                        }

                        $this->disk->put($finalPath, $encoded);
                    } catch (\Exception $e) {
                        return response()->json([
                            'error' => 'Image processing failed: ' . $e->getMessage(),
                            'filename' => $filename
                        ], 500);
                    }
                } else {
                    // Regular file (PDF, DOCX, etc.)
                    $this->disk->putFileAs($path, $file, $filename);
                }

                $uploadedFiles[] = [
                    'name' => $filename,
                    'path' => $filepath,
                    'url' => $this->disk->url($filepath)
                ];
            }

            return response()->json([
                'message' => 'Files uploaded successfully',
                'files' => $uploadedFiles
            ]);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Upload failed: ' . $e->getMessage()
            ], 500);
        }
    }

    /**
     * Delete multiple items
     */
    public function deleteMultiple(Request $request)
    {
        $request->validate(['paths' => 'required|array']);

        foreach ($request->paths as $path) {
            $sanitized = $this->sanitizePath($path);
            $absolutePath = $this->disk->path($sanitized);

            if ($this->disk->exists($sanitized)) {
                if (is_dir($absolutePath)) {
                    $this->disk->deleteDirectory($sanitized);
                } else {
                    $this->disk->delete($sanitized);
                }
            }
        }

        return response()->json(['message' => 'Selected items deleted']);
    }

    /**
     * Delete a folder
     */
    public function deleteFolder(Request $request)
    {
        $request->validate(['path' => 'required|string']);

        $path = $this->sanitizePath($request->input('path'));
        $absolutePath = $this->disk->path($path);

        if ($this->disk->exists($path) && is_dir($absolutePath)) {
            $this->disk->deleteDirectory($path);

            return response()->json(['message' => 'Folder deleted']);
        }

        return response()->json([
            'error' => 'Folder not found or not a directory',
            'path' => $path
        ], 404);
    }

    /**
     * Delete a single file.
     */
    public function delete(Request $request)
    {
        $request->validate([
            'filename' => 'required|string',
        ]);

        $relativePath = $this->sanitizePath($request->input('filename'));
        $fullPath = $this->disk->path($relativePath);

        if (!$this->disk->exists($relativePath)) {
            return response()->json(['error' => 'File not found'], 404);
        }

        try {
            $this->disk->delete($relativePath);

            return response()->json(['message' => 'File deleted successfully']);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Delete failed',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Rename an item
     */
    public function rename(Request $request)
    {
        $request->validate([
            'old_name' => 'required|string',
            'new_name' => 'required|string',
            'path' => 'required|string'
        ]);

        $basePath = $this->sanitizePath($request->input('path'));
        $oldName = $request->input('old_name');
        $newName = $request->input('new_name');

        $oldPath = $basePath . '/' . $oldName;

        // Get absolute path to detect if it's a directory
        $absoluteOldPath = $this->disk->path($oldPath);
        $isDirectory = is_dir($absoluteOldPath);

        // Only check extensions if this is a file
        if (!$isDirectory) {
            $oldExt = pathinfo($oldName, PATHINFO_EXTENSION);
            $newExt = pathinfo($newName, PATHINFO_EXTENSION);
            if (strtolower($oldExt) !== strtolower($newExt)) {
                return response()->json([
                    'error' => 'Changing file extension is not allowed.'
                ], 422);
            }
        }

        // Sanitize name
        $sanitizedNewName = $isDirectory
            ? $this->sanitizeFolderName($newName)
            : $this->sanitizeFilename($newName);

        $newPath = $basePath . '/' . $sanitizedNewName;

        if (!$this->disk->exists($oldPath)) {
            return response()->json([
                'error' => 'Original item does not exist',
                'path' => $oldPath
            ], 404);
        }

        if ($this->disk->exists($newPath)) {
            return response()->json([
                'error' => 'Target already exists',
                'path' => $newPath
            ], 422);
        }

        try {
            $this->disk->move($oldPath, $newPath);

            return response()->json(['message' => 'Renamed successfully']);
        } catch (\Exception $e) {
            return response()->json([
                'error' => 'Rename failed internally',
                'details' => $e->getMessage()
            ], 500);
        }
    }

    /**
     * Create a new folder
     */
    public function createFolder(Request $request)
    {
        $validator = Validator::make($request->all(), [
            'path' => 'required|string',
            'name' => 'required|string'
        ]);

        if ($validator->fails()) {
            return response()->json(['errors' => $validator->errors()], 422);
        }

        $path = $this->sanitizePath($request->input('path'));
        $name = $this->sanitizeFolderName($request->input('name'));
        $newPath = $path . '/' . $name;

        if ($this->disk->exists($newPath)) {
            return response()->json(['error' => 'Folder already exists'], 422);
        }

        $this->disk->makeDirectory($newPath);

        return response()->json([
            'message' => 'Folder created successfully',
            'path' => $newPath
        ]);
    }

    /**
     * Render the TinyMCE integration view.
     */
    public function tinymce()
    {
        return template('media-manager/tinymce.twig', [
            'config' => [
                'routes' => [
                    'browse' => route('mediamanager.browse'),
                    'upload' => route('mediamanager.upload'),
                    'delete' => route('mediamanager.delete'),
                    'createFolder' => route('mediamanager.create-folder'),
                    'download' => route('mediamanager.download'),
                ],
                'csrfToken' => csrf_token(),
                'maxFileSize' => $this->config['max_file_size'] ?? (10 * 1024 * 1024),
                'maxUploadFiles' => $this->config['max_upload_files'] ?? 10,
                'allowedExtensions' => $this->config['allowed_extensions'] ?? [
                        'jpg',
                        'jpeg',
                        'png',
                        'gif',
                        'pdf',
                        'docx',
                        'xlsx'
                    ],
                'app_name' => config('app.name')
            ]
        ]);
    }

    /**
     * Sanitize a given path to prevent directory traversal and normalize slashes.
     */
    protected function sanitizePath($path)
    {
        // Strip out any traversal attempts
        $path = str_replace(['..', '\\'], '', $path);

        // Normalize redundant slashes
        $path = preg_replace('#/+#', '/', $path);

        // Remove leading/trailing slashes and return as absolute-style path
        return '/' . trim($path, '/');
    }

    /**
     * Sanitize a filename by slugifying the base name while preserving the extension.
     */
    protected function sanitizeFilename($filename)
    {
        $extension = pathinfo($filename, PATHINFO_EXTENSION);
        $name = pathinfo($filename, PATHINFO_FILENAME);

        return Str::slug($name) . '.' . $extension;
    }

    /**
     * Sanitize a folder name to a URL-safe slug.
     */
    protected function sanitizeFolderName($name)
    {
        return Str::slug($name);
    }

    /**
     * Force-download any file by path.
     */
    public function download(Request $request)
    {
        // sanitize & normalize
        $path = $this->sanitizePath($request->query('path', '/'));

        // 404 if missing
        if (! $this->disk->exists($path)) {
            abort(404, 'File not found');
        }

        // get on-disk absolute path (works for local disks)
        $absolute = $this->disk->path($path);

        // this sends Content‑Disposition: attachment
        return response()->download($absolute, basename($path));
    }
}