<?php

namespace Mtc\MediaManager\Providers;

use Illuminate\Support\ServiceProvider;
use Illuminate\Support\Facades\Gate;
use Illuminate\Support\Facades\App;

class MediaManagerServiceProvider extends ServiceProvider
{
    /**
     * Register services.
     */
    public function register(): void
    {
        $this->mergeConfigFrom(
            __DIR__ . '/../../config/mediamanager.php', 'mediamanager'
        );
    }

    /**
     * Bootstrap services.
     */
    public function boot(): void
    {
        // Dynamically register the 'media' disk config using values from the config
        $mediaPath = config('mediamanager.media_path', base_path('uploads/media'));
        $mediaUrl  = config('mediamanager.media_url', env('APP_URL') . '/uploads/media');

        config([
            'filesystems.disks.media' => [
                'driver' => 'local',
                'root' => $mediaPath,
                'url' => $mediaUrl,
                'visibility' => 'public',
            ],
        ]);

        // Register package routes
        $this->loadRoutesFrom(__DIR__ . '/../../routes/web.php');

        // Register package views for Twig
        $this->registerTwigTemplatePath();

        // Only run publishing and config setup when in console (e.g., during install or setup)
        if ($this->app->runningInConsole()) {
            // Ensure the media folder exists
            if (!file_exists($mediaPath)) {
                mkdir($mediaPath, 0755, true);
            }

            // publish asset and configuration files
            $this->publishes([
                __DIR__ . '/../../resources/stubs/.htaccess.tmp' => $mediaPath . '/.htaccess',
                __DIR__ . '/../../config/mediamanager.php' => config_path('mediamanager.php'),
                __DIR__ . '/../../resources/css' => public_path('vendor/media-manager/css'),
                __DIR__ . '/../../resources/js'  => public_path('vendor/media-manager/js'),
            ], 'media-manager');
        }

        // Register a gate to control access to the media manager
        Gate::define('access-media-manager', static function ($user) {
            return $user->isAdmin();
        });
    }

    /**
     * @return void
     */
    protected function registerTwigTemplatePath(): void
    {
        // Pharmacy platform uses custom twig template loaders.
        $view_paths = \Config::get('view.paths');
        $view_paths[] = dirname(__DIR__, 2) . '/resources/templates';
        \Config::set('view.paths', $view_paths);
    }
} 