window.mediaManager = function () {
    return {
        files: [],
        folders: [],
        currentPath: '/',
        lastFetchedPath: null,
        breadcrumb: [],
        search: '',
        selectedItems: [],
        loading: false,
        sortField: 'name',
        sortDirection: 'asc',
        maxUploadFiles: window.mediaManagerConfig?.maxUploadFiles || 50,
        dragging: false,
        copiedUrl: null,
        maxFileSize: window.mediaManagerConfig?.maxFileSize || 10485760,
        allowedExtensions: window.mediaManagerConfig?.allowedExtensions || ['jpg', 'jpeg', 'png', 'pdf'],
        successMessage: '',
        errorMessage: '',
        toast: {
            message: '',
            type: '',
            show: false,
            timeout: null
        },
        previewUrl: null,
        showPreview: false,
        previewIndex: null,
        previewImages: [],
        isTinyMCE: window.location.pathname.includes('/tinymce'),
        _lastSortField: null,
        _lastSortDirection: null,
        _lastSearch: '',
        _cachedSortedFiles: null,
        _cachedSortedFolders: null,
        _lastFolderSortField: null,
        _lastFolderSortDirection: null,
        uploadProgress: 0,
        uploading: false,

        get currentPreview() {
            return this.previewImages[this.previewIndex] || null;
        },

        showToast(message, type = 'success') {
            this.toast.message = message;
            this.toast.type = type;
            this.toast.show = true;
            clearTimeout(this.toast.timeout);
            this.toast.timeout = setTimeout(() => this.toast.show = false, 3000);
        },

        showError(message) {
            this.showToast(message, 'error');
        },

        navigateTo(path) {
            this.currentPath = this.sanitizePath(path);
            this.fetchFiles();
        },

        goToCrumb(index) {
            this.currentPath = index === -1 ? '/' : '/' + this.breadcrumb.slice(0, index + 1).join('/');
            this.fetchFiles();
        },

        sanitizePath(path) {
            return path.replace(/^\/+/g, '').replace(/\/+$/g, '');
        },

        formatSize(bytes) {
            if (!bytes) return '-';
            const sizes = ['B', 'KB', 'MB', 'GB'];
            const i = Math.floor(Math.log(bytes) / Math.log(1024));
            return (bytes / Math.pow(1024, i)).toFixed(1) + ' ' + sizes[i];
        },

        get filteredFiles() {
            return this.files.filter(f => f.name.toLowerCase().includes(this.search.toLowerCase()));
        },

        get filteredFolders() {
            return this.folders.filter(f => f.name.toLowerCase().includes(this.search.toLowerCase()));
        },

        fetchFiles() {
            this.loading = true;
            fetch(`/admin/media-manager/browse?path=${encodeURIComponent(this.currentPath)}`)
                .then(r => r.json())
                .then(data => {
                    this.folders = data.items.filter(i => i.type === 'dir');
                    this.files = data.items
                        .filter(i => i.type === 'file')
                        .map(file => this.enhanceFile(file));

                    this._cachedSortedFolders = null;
                    this._cachedSortedFiles = null;

                    this.currentPath = data.current_path || '/';
                    this.breadcrumb = this.currentPath.split('/').filter(Boolean);
                    this.selectedItems = [];
                })
                .catch(() => this.showError('Failed to fetch files.'))
                .finally(() => this.loading = false);
        },

        handleDrop(e) {
            this.uploadFiles(e.dataTransfer.files);
        },

        handleSelect(e) {
            this.uploadFiles(e.target.files);
        },

        changeFolder(path) {
            if (this.currentPath !== path) {
                this.currentPath = path;
                this.fetchFiles();
            }
        },

        copyToClipboard(url) {
            navigator.clipboard.writeText(url)
                .then(() => {
                    this.copiedUrl = url;
                    this.showToast('URL copied to clipboard');
                    setTimeout(() => this.copiedUrl = null, 1500);
                })
                .catch(() => this.showError('Failed to copy URL.'));
        },

        renameItem(item) {
            const newName = prompt('Rename to:', item.name);
            if (!newName || newName === item.name) return;

            if (item.type !== 'dir') {
                const oldExt = item.name.split('.').pop().toLowerCase();
                const newExt = newName.split('.').pop().toLowerCase();
                if (oldExt !== newExt) return this.showError('You cannot change the file extension.');
            }

            this.loading = true;
            fetch('/admin/media-manager/rename', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                },
                body: JSON.stringify({old_name: item.name, new_name: newName, path: this.currentPath})
            })
                .then(async r => {
                    const d = await r.json();
                    if (r.ok) {
                        this.showToast('Renamed successfully');
                        this.fetchFiles();
                    } else this.showError(d.error || 'Rename failed.');
                })
                .catch(() => this.showError('Network error during rename.'))
                .finally(() => this.loading = false);
        },

        deleteFile(path) {
            if (!confirm(`Delete ${path}?`)) return;

            this.loading = true;
            fetch('/admin/media-manager/delete', {
                method: 'DELETE',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                },
                body: JSON.stringify({filename: path}) // use full path here
            })
                .then(async r => {
                    const d = await r.json();
                    if (r.ok) {
                        this.showToast('File deleted');
                        this.fetchFiles();
                    } else this.showError(d.error || 'Failed to delete file.');
                })
                .catch(() => this.showError('Network error during deletion.'))
                .finally(() => this.loading = false);
        },

        deleteFolder(path) {
            if (!confirm(`Delete folder ${path}?`)) return;
            this.loading = true;
            fetch('/admin/media-manager/delete-folder', {
                method: 'DELETE',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                },
                body: JSON.stringify({path})
            })
                .then(async r => {
                    const d = await r.json();
                    if (r.ok) {
                        this.showToast('Folder deleted');
                        this.fetchFiles();
                    } else this.showError(d.error || 'Failed to delete folder.');
                })
                .catch(() => this.showError('Network error during folder deletion.'))
                .finally(() => this.loading = false);
        },

        promptCreateFolder() {
            const name = prompt('Enter new folder name:');
            if (!name) return;
            this.loading = true;
            fetch('/admin/media-manager/create-folder', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                },
                body: JSON.stringify({name, path: this.currentPath})
            })
                .then(async r => {
                    const d = await r.json();
                    if (r.ok) {
                        this.showToast('Folder created');
                        this.fetchFiles();
                    } else if (d.errors) this.showError(Object.values(d.errors).flat().join(', '));
                    else this.showError(d.error || 'Failed to create folder.');
                })
                .catch(() => this.showError('Network error while creating folder.'))
                .finally(() => this.loading = false);
        },

        selectItem(item) {
            const id = item.path;

            const isSelected = this.selectedItems.some(i => i.path === id);

            if (isSelected) {
                // Deselect
                this.selectedItems = this.selectedItems.filter(i => i.path !== id);
            } else {
                // Select
                this.selectedItems = [...this.selectedItems, item];
            }
        },

        deleteSelected() {
            if (!confirm('Delete selected items?')) return;
            this.loading = true;
            fetch('/admin/media-manager/delete-multiple', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                },
                body: JSON.stringify({ paths: this.selectedItems.map(i => i.path) })
            })
                .then(async r => {
                    const d = await r.json();
                    if (r.ok) {
                        this.showToast('Selected items deleted');
                        this.fetchFiles();
                    } else if (d.errors) this.showError(Object.values(d.errors).flat().join(', '));
                    else this.showError(d.error || 'Failed to delete selected items.');
                })
                .catch(() => this.showError('Network error while deleting selected items.'))
                .finally(() => this.loading = false);
        },

        uploadFiles(fileList) {
            const maxCombinedSize = this.maxFileSize * this.maxUploadFiles;
            const totalSize = [...fileList].reduce((sum, f) => sum + f.size, 0);

            if (fileList.length > this.maxUploadFiles) {
                this.showToast(`You can only upload up to ${this.maxUploadFiles} files at a time.`, 'error');
                return;
            }

            if (totalSize > maxCombinedSize) {
                this.showToast(`Combined file size exceeds ${Math.floor(maxCombinedSize / (1024 * 1024))}MB limit.`, 'error');
                return;
            }

            const formData = new FormData();
            for (const file of fileList) {
                formData.append('files[]', file);
            }
            formData.append('path', this.currentPath);

            this.uploadProgress = 0;
            this.uploading = true;
            this.loading = true;

            const xhr = new XMLHttpRequest();
            xhr.open('POST', '/admin/media-manager/upload', true);
            xhr.setRequestHeader('X-CSRF-TOKEN', document.querySelector('meta[name="csrf-token"]').content);

            xhr.upload.addEventListener('progress', e => {
                if (e.lengthComputable) {
                    this.uploadProgress = Math.round((e.loaded / e.total) * 100);
                }
            });

            xhr.onload = async () => {
                this.uploading = false;
                this.loading = false;

                if (xhr.status === 413) {
                    this.showToast('Upload failed: File(s) too large or total size exceeded.', 'error');
                    return;
                }

                let d;
                try {
                    d = JSON.parse(xhr.responseText);
                } catch {
                    this.showToast('Invalid server response.', 'error');
                    return;
                }

                if (xhr.status >= 200 && xhr.status < 300) {
                    this.showToast('Files uploaded successfully.');
                    this.fetchFiles();
                } else {
                    this.showToast(d.error || 'Upload failed.', 'error');
                }
            };

            xhr.onerror = () => {
                this.uploading = false;
                this.loading = false;
                this.showToast('Network error during upload.', 'error');
            };

            xhr.send(formData);
        },

        fileIcon(name) {
            const ext = name.split('.').pop().toLowerCase();
            return ['pdf', 'docx', 'xlsx', 'csv'].includes(ext) ? '📄' : '📁';
        },

        getFileIcon(name) {
            const ext = name.split('.').pop().toLowerCase();
            switch (ext) {
                case 'pdf': return 'mdi:file-pdf-box';
                case 'docx': return 'mdi:file-word-box';
                case 'xlsx': return 'mdi:file-excel-box';
                case 'csv': return 'mdi:file-delimited';
                case 'zip':
                case 'rar': return 'mdi:folder-zip';
                default: return 'mdi:file';
            }
        },

        getFileColor(name) {
            const ext = name.split('.').pop().toLowerCase();
            switch (ext) {
                case 'pdf': return 'text-red-500';
                case 'docx': return 'text-blue-600';
                case 'xlsx':
                case 'csv': return 'text-green-600';
                case 'zip':
                case 'rar': return 'text-yellow-600';
                default: return 'text-gray-500';
            }
        },

        formatTimestamp(timestamp) {
            if (!timestamp) return '—';
            const date = new Date(timestamp * 1000);
            return date.toLocaleDateString(undefined, {
                year: 'numeric', month: 'short', day: 'numeric', hour: '2-digit', minute: '2-digit'
            });
        },

        sortBy(field) {
            this.sortDirection = this.sortField === field && this.sortDirection === 'asc' ? 'desc' : 'asc';
            this.sortField = field;
        },

        get sortedFiles() {
            const search = this.search.toLowerCase();

            // Check if cache is valid
            const cacheValid =
                this._cachedSortedFiles &&
                this._lastSortField === this.sortField &&
                this._lastSortDirection === this.sortDirection &&
                this._lastSearch === search;

            if (cacheValid) {
                return this._cachedSortedFiles;
            }

            // Filter and sort
            const filtered = this.files.filter(f =>
                f.name.toLowerCase().includes(search)
            );

            const sorted = [...filtered].sort((a, b) => {
                const aVal = a[this.sortField];
                const bVal = b[this.sortField];

                if (aVal < bVal) return this.sortDirection === 'asc' ? -1 : 1;
                if (aVal > bVal) return this.sortDirection === 'asc' ? 1 : -1;
                return 0;
            });

            // Cache result
            this._lastSortField = this.sortField;
            this._lastSortDirection = this.sortDirection;
            this._lastSearch = search;
            this._cachedSortedFiles = sorted;

            return sorted;
        },

        get sortedFolders() {
            const cacheValid =
                this._cachedSortedFolders &&
                this._lastFolderSortField === this.sortField &&
                this._lastFolderSortDirection === this.sortDirection;

            if (cacheValid) {
                return this._cachedSortedFolders;
            }

            const sorted = [...this.folders].sort((a, b) => {
                const aVal = a[this.sortField];
                const bVal = b[this.sortField];

                if (aVal < bVal) return this.sortDirection === 'asc' ? -1 : 1;
                if (aVal > bVal) return this.sortDirection === 'asc' ? 1 : -1;
                return 0;
            });

            this._lastFolderSortField = this.sortField;
            this._lastFolderSortDirection = this.sortDirection;
            this._cachedSortedFolders = sorted;

            return sorted;
        },

        previewImage(url, name) {
            const allImages = [...this.sortedFiles].filter(f => f.mime && f.mime.startsWith('image'));
            this.previewImages = allImages;
            this.previewIndex = allImages.findIndex(f => f.url === url);
            this.showPreview = true;
        },

        insertIntoTinyMCE(url) {
            try {
                const origin = window.location.origin;
                const pathOnly = url.replace(origin, '');

                if (window.opener && typeof window.opener.insertMediaToTinyMCE === 'function') {
                    window.opener.insertMediaToTinyMCE(pathOnly);
                    window.close();
                } else {
                    this.showToast('Unable to insert into editor. No callback found.', 'error');
                }
            } catch (e) {
                this.showToast('Invalid file URL.', 'error');
            }
        },

        getRowClass(item) {
            return this.selectedItems.some(i => i.path === item.path)
                ? 'bg-blue-100 shadow-[inset_4px_0_0_0_rgb(59,130,246)]'
                : 'group hover:bg-gray-100 hover:shadow-sm';
        },

        enhanceFile(file) {
            const ext = file.name.split('.').pop().toLowerCase();
            file.extLabel = '.' + ext;
            file.sizeLabel = this.formatSize(file.size);
            file.modifiedLabel = this.formatTimestamp(file.modified);
            file.icon = this.getFileIcon(file.name);
            file.iconColor = this.getFileColor(file.name);
            file.isImage = file.mime?.startsWith('image');
            return file;
        },

        init() {
            this.fetchFiles();
        }
    };
};
