<?php

namespace Mtc\PayPalPayments;

use GuzzleHttp\Exception\ClientException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Log;
use Mtc\Checkout\Contracts\HandlesRefunds;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\PaymentGateway;
use Mtc\Checkout\Invoice\Payment;
use Mtc\Checkout\PaymentForm;
use Mtc\PayPalPayments\Services\PaymentService;

class PayPal implements PaymentGateway, HandlesRefunds
{
    /**
     * @var PaymentService
     */
    protected $service;

    /**
     * @var PayPalSettings
     */
    protected $settings;

    public function __construct(PaymentService $service, PayPalSettings $settings)
    {
        $this->service = $service;
        $this->settings = $settings;
    }

    /**
     * @param InvoiceRepositoryContract $invoice
     * @param \Mtc\Checkout\Contracts\PayableContract $payable
     * @return bool
     */
    public function isApplicable(InvoiceRepositoryContract $invoice, $payable): bool
    {
        if (config('paypal_payments.enabled') === false || $this->isPayPalLinkedUp() === false) {
            return false;
        }

        if ($invoice->getOutstandingAmount() <= config('paypal_payments.minimal_transaction_amount')) {
            return false;
        }

        $settings = App::make(PayPalSettings::class);
        if (empty($settings->get('API_CLIENT_ID')) || empty($settings->get('API_CLIENT_ID'))) {
            return false;
        }

        return App::make(config('paypal_payments.applicable_check_class'))->handle($invoice, $payable);
    }

    /**
     * @param InvoiceRepositoryContract $invoice
     * @return PaymentForm
     */
    public function getPaymentForm(InvoiceRepositoryContract $invoice): PaymentForm
    {
        return new PaymentForm('paypal-payments', 'vue-component', [
            'invoice_id' => $invoice->getId(),
            'name' => __('paypal_payments::paypal.button_payment_option_name'),
            'layout' => $this->settings->get('button_layout'),
            'shape' => $this->settings->get('button_shape'),
            'colour' => $this->settings->get('button_colour'),
            'tagline' => $this->settings->get('button_tagline') ?? 'true',
            'size' => $this->settings->get('button_size'),
            'label' => $this->settings->get('button_label'),
        ]);
    }

    /**
     * Done via internal controller
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return array
     */
    public function charge(Request $request, InvoiceRepositoryContract $invoice): array
    {
        return [];
    }

    /**
     * @param Payment $payment
     * @return bool
     */
    public function isRefundable(Payment $payment)
    {
        return !empty($payment->details['capture_id']);
    }

    /**
     * @param Payment $payment
     * @param null $amount
     * @return array
     */
    public function refund(Payment $payment, $amount = null)
    {
        try {
            $response = $this->service->refund(
                $payment->details['capture_id'],
                $amount,
                $payment->currency_code,
                'REFUND-' . $payment->invoice_id . '-' . $payment->id
            );
            if ($response['status'] === 'COMPLETED') {
                return [
                    'reference' => $response['id'],
                    'amount' => $amount,
                ];
            }
            if ($response['status'] === 'PENDING') {
                return [
                    'reference' => $response['id'],
                    'amount' => $amount,
                    'details' => [
                        'Status' => 'Pending',
                        'Reason' => $response['status_details']['reason'],
                    ]
                ];
            }
        } catch (ClientException $exception) {
            $context = (string)$exception->getResponse()->getBody();

            Log::warning('Failed Refunding Payment [PayPal]', [
                'payment' => $payment,
                'exception' => $context,
            ]);
            $context = json_decode($context);
            if (isset($context->details[0]->description)) {
                $invoice = App::make(InvoiceRepositoryContract::class)->load($payment->invoice_id);

                $details = $invoice->details;
                $details['note'] = $context->message . ': ' . $context->details[0]->description;
                $invoice->details = $details;
                $invoice->save();
            }
        } catch (\Exception $exception) {
            Log::warning('Failed Refunding Payment [PayPal]', [
                'payment' => $payment,
                'exception' => $exception,
            ]);
        }
        return [];
    }

    /**
     * @return bool
     */
    protected function isPayPalLinkedUp(): bool
    {
        return App::make(PayPalSettings::class)
            ->allowPayments();
    }
}
