<?php

namespace Mtc\PayPalPayments\Services;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\ClientException;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Config;
use Mtc\PayPalPayments\PayPalSettings;

class PayPalMicroServiceApi
{
    /**
     * @var Client
     */
    protected $api;

    public function __construct(Client $api)
    {
        $this->api = $api;
    }

    /**
     * @param string $site_name
     * @param string $site_domain
     * @return mixed
     */
    public function account(string $site_name, string $site_domain)
    {
        $result = $this->api->post($this->endpoint('register-customer'), [
            'headers' => [
                'Content-Type' => 'application/json',
                'Accept' => 'application/json',
            ],
            'body' => json_encode([
                'url' => $site_domain,
                'name' => $site_name,
                'notify_url' => route('paypal-service-webhook'),
            ]),
        ]);
        return json_decode((string)$result->getBody());
    }

    /**
     * @param $path
     * @return mixed
     */
    public function get($path)
    {
        $result = $this->api->get($this->endpoint($path), [
            'headers' => [
                'Content-Type' => 'application/json',
            ],
        ]);
        return json_decode((string)$result->getBody());
    }

    /**
     * @param $path
     * @param array $data
     * @return mixed
     */
    public function post($path, array $data = [])
    {
        $data['api_token'] = $this->getAccessToken();
        $context = [
            'headers' => [
                'Content-Type' => 'application/json',
            ],
        ];

        if (!empty($data)) {
            $context['body'] = json_encode($data);
        }

        $result = $this->api->post($this->endpoint($path), $context);
        return json_decode((string)$result->getBody(), true);
    }

    /**
     * @param PayPalSettings $settings
     */
    public function ensureApiCredentials(PayPalSettings $settings)
    {
        if ($settings->get('API_CLIENT_ID') !== null) {
            return;
        }

        $response = $this->api->post($this->endpoint('api-keys'), [
            'headers' => [
                'X-Auth' => $this->authKey()
            ]
        ]);
        $result = json_decode((string)$response->getBody(), true);

        if (isset($result['client_id'])) {
            $settings->save('API_CLIENT_ID', $result['client_id']);
            $settings->save('API_CLIENT_SECRET', $result['client_secret']);
        }
    }

    /**
     * @param $path
     * @return string
     */
    protected function endpoint($path): string
    {
        return 'https://www.paypal.mtcassets.com/api/' . ltrim($path, '/');
    }

    /**
     * @return string
     */
    private function getAccessToken(): string
    {
        /** @var PayPalSettings $settings */
        $settings = App::make(PayPalSettings::class);
        return $settings->get('PP_SERVICE_API_KEY');
    }

    /**
     * @return string
     */
    private function authKey()
    {
        return sha1('mtc-' . config('paypal_payments.partner_id'));
    }
}
