<?php

namespace Mtc\PayPalPayments\Http\Controllers;

use GuzzleHttp\Exception\ClientException;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\URL;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\PayPalPayments\Services\PaymentService;

class PayPalPaymentController
{
    /**
     * Create Order
     *
     * @param $invoice_id
     * @param InvoiceRepositoryContract $invoice
     * @param PaymentService $service
     * @return mixed|string[]
     */
    public function create($invoice_id, InvoiceRepositoryContract $invoice, PaymentService $service)
    {
        try {
            $invoice->load($invoice_id);
            $order_data = $service->createOrder($invoice);
            $this->attachOrderDataToInvoice($invoice, $order_data);
            return $order_data;
        } catch (ClientException $exception) {
            Log::error(
                'Failed to create order',
                [
                    'message' => $exception->getMessage(),
                    'body' => (string)$exception->getResponse()->getBody()
                ]
            );
        } catch (\Exception $exception) {
            Log::error('Failed to create order', [$exception->getMessage()]);
        }
        return [
            'error' => 'Failed to create order'
        ];
    }

    /**
     * Approve Order/Payment
     *
     * @param Request $request
     * @param $invoice_id
     * @param InvoiceRepositoryContract $invoice
     * @param PaymentService $service
     * @return array
     */
    public function approve(Request $request, $invoice_id, InvoiceRepositoryContract $invoice, PaymentService $service)
    {
        try {
            $response = $service->capture($request->input('paypal_order_id'));
        } catch (ClientException $exception) {
            return $this->softFailure($exception);
        } catch (\Exception $exception) {
            return $this->hardFailure($exception);
        }

        $invoice->load($invoice_id);

        $invoice->savePayment([
            'provider' => 'PayPal',
            'amount' => $response['amount'],
            'currency_code' => $response['currency_code'],
            'reference' => $response['id'],
            'details' => $response,
            'confirmed_at' => now(),
        ]);

        $is_live = config('app.env') === 'production';
        return [
            'success' => true,
            'redirect' => URL::signedRoute('successful_payment', ['id' => $invoice->getId()], null, $is_live),
        ];
    }

    /**
     * Capture Payment (3D secure/card)
     *
     * @param Request $request
     * @param $invoice_id
     * @param InvoiceRepositoryContract $invoice
     * @param PaymentService $service
     * @return array
     */
    public function capture(Request $request, $invoice_id, InvoiceRepositoryContract $invoice, PaymentService $service)
    {
        try {
            if ($service->verify3DSecure($service->getOrder($request->input('orderID'))) !== true) {
                return [
                    'success' => false,
                    'message' => 'Failed to authenticate payment. Please try again',
                ];
            }
            $response = $service->capture($request->input('orderID'));
        } catch (ClientException $exception) {
            return $this->softFailure($exception);
        } catch (\Exception $exception) {
            return $this->hardFailure($exception);
        }

        $invoice->load($invoice_id);

        $invoice->savePayment([
            'provider' => 'PayPal',
            'amount' => $response['amount'],
            'currency_code' => $response['currency_code'],
            'reference' => $response['id'],
            'details' => $response,
            'confirmed_at' => now(),
        ]);

        $is_live = config('app.env') === 'production';
        return [
            'success' => true,
            'redirect' => URL::signedRoute('successful_payment', ['id' => $invoice->getId()], null, $is_live),
        ];
    }

    /**
     * Hard failure response
     *
     * @param \Exception $exception
     * @return array
     */
    public function hardFailure(\Exception $exception): array
    {
        return [
            'success' => false,
            'message' => $exception->getMessage(),
        ];
    }

    /**
     * Soft failure response
     *
     * @param ClientException $exception
     * @return array
     */
    public function softFailure(ClientException $exception): array
    {
        $exception_data = json_decode((string)$exception->getResponse()->getBody());
        return [
            'success' => false,
            'message' => $exception_data->details[0]->description,
            'error_type' => $exception_data->details[0]->issue,
        ];
    }

    /**
     * @param Request $request
     * @param $invoice_id
     * @param InvoiceRepositoryContract $invoice
     */
    public function notify(Request $request, $invoice_id, InvoiceRepositoryContract $invoice)
    {
        $invoice->load($invoice_id);
        Log::info('PayPal Payment Notification', $request->input());
    }

    /**
     * Attach PayPal Order ID to Invoice
     *
     * @param $invoice
     * @param $order_data
     */
    protected function attachOrderDataToInvoice($invoice, $order_data)
    {
        $invoice_model = $invoice->getModel();
        $details = $invoice_model->details ?? [];
        $details['paypal_order_id'] = $order_data['id'];
        $invoice_model->details = $details;
        $invoice_model->save();
    }
}
