<?php

namespace Mtc\PayPalPayments\Http\Controllers;

use Illuminate\Contracts\Foundation\Application;
use Illuminate\Contracts\View\Factory;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Log;
use Illuminate\View\View;
use Mtc\Foundation\Country;
use Mtc\PayPalPayments\Http\Requests\PayPalSettingRequest;
use Mtc\PayPalPayments\PayPalSettings;
use Mtc\PayPalPayments\Services\PayPalMicroServiceApi;
use Mtc\PayPalPayments\Services\ReferralSignupService;

class PayPalConfigController
{
    /**
     * @param PayPalSettings $config
     * @param PayPalMicroServiceApi $microservice
     * @return Application|Factory|View
     */
    public function index(PayPalSettings $config, PayPalMicroServiceApi $microservice)
    {
        $microservice->ensureApiCredentials($config);

        return view('paypal::config.index', [
            'config' => $config,
            'countries' => Country::query()->select('code', 'name')->get(),
        ]);
    }

    /**
     * @param PayPalSettingRequest $request
     * @param ReferralSignupService $signup_service
     * @return array
     */
    public function createConsent(PayPalSettingRequest $request, ReferralSignupService $signup_service)
    {
        try {
            return [
                'action' => 'redirect',
                'url' => $signup_service->createReferral($request)
            ];
        } catch (\Exception $exception) {
            return [
                'action' => 'error',
                'message' => $exception->getMessage(),
            ];
        }
    }

    /**
     * @param ReferralSignupService $signup_service
     * @param PayPalSettings $settings
     * @return \Illuminate\Http\RedirectResponse
     */
    public function refresh(ReferralSignupService $signup_service, PayPalSettings $settings)
    {
        try {
            $merchant_status = $signup_service->getMerchantStatus($settings->get('merchant_id'));
            $settings->saveMerchantStatus($merchant_status);
            session()->flash('success', 'Details updated');
        } catch (\Exception $exception) {
            Log::error('Failed to update merchant status', [$exception]);
            session()->flash('error', 'Failed to update details. Try again later');
        }
        return back();
    }

    /**
     * @param ReferralSignupService $signup_service
     * @param PayPalSettings $settings
     * @return \Illuminate\Http\RedirectResponse
     */
    public function refreshConsent(ReferralSignupService $signup_service, PayPalSettings $settings)
    {
        try {
            $result = $signup_service->getSignupStatus($settings->get('SIGNUP_TRACKING_ID'));
            if (empty($result->merchant_id)) {
                session()->flash('success', 'Unable to retrieve details. Try signing up again');
                return back();
            }

            $signup_service->attachMerchantId($result->merchant_id);
            $merchant_status = $signup_service->getMerchantStatus($settings->get('merchant_id'));
            $settings->saveMerchantStatus($merchant_status);
            session()->flash('success', 'Details updated');
        } catch (\Exception $exception) {
            Log::error('Failed to update merchant status', [$exception]);
            session()->flash('error', 'Failed to update details. Try again later');
        }
        return back();
    }

    /**
     * @param Request $request
     * @param ReferralSignupService $signup_service
     * @param PayPalSettings $settings
     * @return Application|\Illuminate\Http\RedirectResponse|\Illuminate\Routing\Redirector
     */
    public function store(Request $request, ReferralSignupService $signup_service, PayPalSettings $settings)
    {
        try {
            if ($request->filled('merchantIdInPayPal')) {
                $signup_service->attachMerchantId($request->input('merchantIdInPayPal'));

                $merchant_status = $signup_service->getMerchantStatus($request->input('merchantIdInPayPal'));
                $settings->saveMerchantStatus($merchant_status);

                session()->flash('success', 'Account Linked');
                return redirect(route('admin.paypal.index'));
            }
            session()->flash('success', 'No merchant ID retrieved');
        } catch (\Exception $exception) {
            Log::error('Failed to save merchant details', [$exception]);
            session()->flash('error', 'Failed to save details. Try again later');
        }
        return redirect(route('admin.paypal.index'));
    }

    /**
     * @param ReferralSignupService $service
     * @param PayPalSettings $config
     * @return \Illuminate\Http\RedirectResponse
     */
    public function disconnect(ReferralSignupService $service, PayPalSettings $config)
    {
        $config->clear();

        session()->flash('success', 'Account revoked');
        return back();
    }

    /**
     * @param Request $request
     * @param PayPalSettings $settings
     * @return \Illuminate\Http\RedirectResponse
     */
    public function customization(Request $request, PayPalSettings $settings)
    {
        $settings->save('button_layout', $request->input('layout'));
        $settings->save('button_colour', $request->input('colour'));
        $settings->save('button_shape', $request->input('shape'));
        $settings->save('button_size', $request->input('size'));
        $settings->save('button_label', $request->input('label'));
        $settings->save('button_tagline', $request->input('tagline'));

        session()->flash('success', 'Settings Updated');
        return back();
    }

    /**
     * @param Request $request
     * @param PayPalSettings $settings
     * @return \Illuminate\Http\RedirectResponse
     */
    public function togglePPCP(Request $request, PayPalSettings $settings)
    {
        $settings->toggle('PPCP_ENABLED');
        session()->flash('success', 'Settings Updated');
        return back();
    }

    /**
     * @param Request $request
     * @param PayPalSettings $settings
     * @return \Illuminate\Http\RedirectResponse
     */
    public function toggle3DSec(Request $request, PayPalSettings $settings)
    {
        $settings->toggle('3DSEC_ENABLED');
        session()->flash('success', 'Settings Updated');
        return back();
    }

    /**
     * @param Request $request
     * @param PayPalSettings $settings
     * @return \Illuminate\Http\RedirectResponse
     */
    public function save3DSec(Request $request, PayPalSettings $settings)
    {
        if ($request->has('reset_defaults')) {
            $settings->clearSettings('3DSEC_');
        } else {
            foreach ($request->input('3d_sec_actions') as $scenario => $action) {
                $settings->save('3DSEC_' . $scenario, $action);
            }
        }

        session()->flash('success', 'Settings Updated');
        return back();
    }
}
