<?php

namespace Mtc\Paypal;

use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\Basket\Contracts\BasketRepositoryInterface;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Paypal\Contracts\PayPalBasketFillInterface;
use Mtc\Paypal\Contracts\PayPalPaymentFactoryInterface;
use Mtc\Paypal\Exceptions\AlreadyPaidException;
use PayPal\Api\Payment;
use PayPal\Api\PaymentExecution;
use PayPal\Rest\ApiContext;

/**
 * Class PayPalService
 *
 * Implements PayPal EC actions
 *
 * @package Mtc\Paypal
 */
class PayPalService
{

    /**
     * @var object|null
     */
    public $context;

    /**
     * PayPalService constructor.
     *
     * @param ApiContext $context
     */
    public function __construct(ApiContext $context)
    {
        $this->context = $context;
    }

    /**
     * Prepare Payment Authorization
     *
     * @param Request $request
     * @throws \Exception
     */
    public function prepareAuthorize(Request $request)
    {
        // Get payment object by passing paymentId
        $payment = Payment::get($request->input('paymentId'), $this->context);
        $transaction = $this->getLocalPaymentByToken($request->input('token'));

        if (!$this->validPayment($payment, $transaction)) {
            throw new \Exception('Payment verification failed');
        }

        $basket = App::make(BasketRepositoryInterface::class);
        return App::make(PayPalBasketFillInterface::class)->fill($basket, $payment->getPayer()->getPayerInfo());
    }

    /**
     * Handle charging customer
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return mixed
     * @throws \Exception
     */
    public function charge(Request $request, InvoiceRepositoryContract $invoice)
    {
        $payment = Payment::get($request->input('payment_id'), $this->context);
        $transaction = $this->getLocalPaymentByToken($request->input('token'));

        if (!$this->validPayment($payment, $transaction)) {
            throw new \Exception('Payment verification failed');
        }

        /** @var PayPalPaymentFactoryInterface $charging_factory */
        $charging_factory = App::make(PayPalPaymentFactoryInterface::class);

        if ($transaction->amount != $invoice->getOutstandingAmount()) {
            $charging_factory->updateTotals($invoice, $payment);
            $transaction->amount = $invoice->getOutstandingAmount();
            $transaction->save();
        }

        if ($this->isPaymentAlreadyExecuted($request->input('token'))) {
            throw new \Exception('Payment verification failed');
        }

        $execution = (new PaymentExecution())->setPayerId($request->input('payer_id'));
        $data =  $charging_factory->chargePayment($payment, $execution, $this->context);
        $data['reference'] = $request->input('token');
        $data['confirmed_at'] = Carbon::now();
        return $data;

    }

    /**
     * Verify payment info is correct
     *
     * @param $payment
     * @param $invoice_payment
     * @throws AlreadyPaidException
     */
    protected function validPayment($payment, $invoice_payment)
    {
        if (! $payment instanceof Payment || !$invoice_payment) {
            return false;
        }

        if ($invoice_payment->is_confirmed) {
            throw new AlreadyPaidException('Payment already paid');
        }

        return true;
    }

    /**
     * Get the payment information from local record
     * 
     * @param $token
     * @return \Illuminate\Database\Eloquent\Builder|\Illuminate\Database\Eloquent\Model|object
     */
    public function getLocalPaymentByToken($token)
    {
        return \Mtc\Checkout\Invoice\Payment::query()
            ->where('reference', $token)
            ->whereNull('confirmed_at')
            ->first();
    }


    /**
     * Check to see if this token already have execute action recorded
     *
     * @return bool|null
     */
    public function isPaymentAlreadyExecuted($token)
    {
        return \Mtc\Checkout\Invoice\Payment::query()
            ->where('reference', $token)
            ->whereNotNull('confirmed_at')
            ->exists();
    }
}
