<?php

namespace Mtc\Paypal;

use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Mtc\Basket\Contracts\BasketRepositoryInterface;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Invoice\Payment as PaymentModel;
use Mtc\Paypal\Contracts\PayPalBasketFillInterface;
use Mtc\Paypal\Contracts\PayPalPaymentFactoryInterface;
use Mtc\Paypal\Exceptions\AlreadyPaidException;
use PayPal\Api\Amount;
use PayPal\Api\Authorization;
use PayPal\Api\Capture;
use PayPal\Api\Payment;
use PayPal\Api\PaymentExecution;
use PayPal\Api\RefundRequest;
use PayPal\Api\Sale;
use PayPal\Rest\ApiContext;

/**
 * Class PayPalService
 *
 * Implements PayPal EC actions
 *
 * @package Mtc\Paypal
 */
class PayPalService
{

    /**
     * @var object|null
     */
    public $context;

    /**
     * PayPalService constructor.
     *
     * @param ApiContext $context
     */
    public function __construct(ApiContext $context)
    {
        $this->context = $context;
    }

    /**
     * Prepare Payment Authorization
     *
     * @param Request $request
     * @throws \Exception
     */
    public function prepareAuthorize(Request $request)
    {
        // Get payment object by passing paymentId
        $payment = Payment::get($request->input('paymentId'), $this->context);
        $transaction = $this->getLocalPaymentFromToken($request->input('token'));

        if (!$this->validPayment($payment, $transaction)) {
            throw new \Exception('Payment verification failed');
        }

        $basket = App::make(BasketRepositoryInterface::class);
        return App::make(PayPalBasketFillInterface::class)->fill($basket, $payment->getPayer()->getPayerInfo());
    }

    /**
     * Handle charging customer
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return mixed
     * @throws \Exception
     */
    public function charge(Request $request, InvoiceRepositoryContract $invoice)
    {
        $pay_pal_payment = Payment::get($request->input('payment_id'), $this->context);
        $payment = $this->getLocalPaymentFromToken($request->input('token'));

        if (!$this->validPayment($pay_pal_payment, $payment)) {
            throw new \Exception('Payment verification failed');
        }

        /** @var PayPalPaymentFactoryInterface $pay_pal_service */
        $pay_pal_service = App::make(PayPalPaymentFactoryInterface::class);

        if ($payment->amount != $invoice->getOutstandingAmount()) {
            $pay_pal_service->updateTotals($invoice, $pay_pal_payment);
            $payment->amount = $invoice->getOutstandingAmount();
            $payment->save();
        }

        if ($this->isPaymentAlreadyExecuted($request->input('token'))) {
            throw new AlreadyPaidException('Payment verification failed');
        }

        $pay_pal_execution = (new PaymentExecution())->setPayerId($request->input('payer_id'));
        $data = $pay_pal_service->chargePayment($pay_pal_payment, $pay_pal_execution, $this->context);

        if (config('checkout.deferred_payments') === true) {
            if (!isset($data['details']['transactions'][0]['related_resources'][0]['authorization']['id'])) {
                throw new \Exception('Failed to retrieve authorization');
            }
            return [
                'provider' => 'paypal',
                'reference' => $request->input('token'),
                'amount' => $invoice->getOutstandingAmount(),
                'details' => [
                    'payer_id' => $request->input('payer_id'),
                    'payment_id' => $request->input('payment_id'),
                    'authorization_id' => $data['details']['transactions'][0]['related_resources'][0]['authorization']['id'],
                ]
            ];
        }

        $data['reference'] = $request->input('token');
        $data['confirmed_at'] = Carbon::now();
        return $data;
    }

    /**
     * Check if payment can be charged
     *
     * @param $payment_id
     */
    public function isChargeable($payment_data)
    {
        $authorization = Authorization::get($payment_data->details['authorization_id'], $this->context);
        $valid_until = Carbon::parse($authorization->valid_until);
        if ($authorization->state === 'expired' && $valid_until > Carbon::now()) {
            return true;
        }

        return in_array($authorization->state, ['pending', 'authorized']);
    }

    /**
     * Attempt charging a deferred payment
     *
     * @param PaymentModel $payment_data
     * @return bool
     */
    public function deferredCharge(PaymentModel $payment_data)
    {
        $authorization = Authorization::get($payment_data->details['authorization_id'], $this->context);
        if ($authorization->state === 'expired') {
            $authorization->reauthorize($this->context);
        }

        $amount = new Amount();
        $amount->setTotal($payment_data->amount);
        $amount->setCurrency($payment_data->invoice->currency);

        $capture = new Capture();
        $capture->setAmount($amount);
        $capture->setIsFinalCapture(true);

        $capture_data = $authorization->capture($capture, $this->context)->toArray();

        if ($capture_data['state'] === 'completed') {
            $capture_data['authorization_id'] = $authorization->id;
            $capture_data['capture_id'] = $capture_data['id'];
            $capture_data['id'] = $payment_data->details['payment_id'];
            $payment_data->details = $capture_data;
            $payment_data->save();
        }
        return $capture_data['state'] === 'completed';
    }

    /**
     * Is payment refundable
     *
     * @param $payment
     * @return bool
     */
    public function isRefundable($payment_data)
    {
        $payment = Payment::get($payment_data->details['id'], $this->context);
        $payment_data->refundable_amount = $this->getRefundableAmount($payment);
        return $payment->state === 'approved' && $payment_data->refundable_amount > 0;
    }

    /**
     * Refund a transaction
     *
     * @param $payment
     * @param $amount
     * @return array
     */
    public function refund($payment, $amount)
    {
        $refund = new RefundRequest();
        if ($payment->amount - $amount >= 0.01) {
            $refund_amount = new Amount();
            $refund_amount->setCurrency($payment->invoice->currency);
            $refund_amount->setTotal($amount);
            $refund->setAmount($refund_amount);
            $refund->setInvoiceNumber($refund->id);
            $refund->setDescription("Refund on {$payment->id}");
        }
        $sale_id = $payment->details['transactions'][0]['related_resources'][0]['sale']['id'];
        if (empty($sale_id)) {
            $capture = Capture::get($payment->details['capture_id'], $this->context);
            $refund = $capture->refundCapturedPayment($refund, $this->context);
        } else {
            $sale = Sale::get($sale_id, $this->context);
            $refund = $sale->refundSale($refund, $this->context);
        }


        return [
            'reference' => $refund->id,
            'amount' => $refund->amount->total,
        ];
    }

    /**
     * Verify payment info is correct
     *
     * @param $payment
     * @param $invoice_payment
     * @throws AlreadyPaidException
     */
    protected function validPayment($payment, $invoice_payment)
    {
        if (! $payment instanceof Payment || !$invoice_payment) {
            return false;
        }

        if ($invoice_payment->is_confirmed) {
            throw new AlreadyPaidException('Payment already paid');
        }

        return true;
    }

    /**
     * Get the payment information from local record
     *
     * @param $token
     * @return \Illuminate\Database\Eloquent\Builder|\Illuminate\Database\Eloquent\Model|object
     */
    protected function getLocalPaymentFromToken($token)
    {
        return \Mtc\Checkout\Invoice\Payment::query()
            ->where('reference', $token)
            ->whereNull('confirmed_at')
            ->first();
    }

    /**
     * Check to see if this token already have execute action recorded
     *
     * @return bool|null
     */
    protected function isPaymentAlreadyExecuted($token)
    {
        return \Mtc\Checkout\Invoice\Payment::query()
            ->where('reference', $token)
            ->whereNotNull('confirmed_at')
            ->exists();
    }

    /**
     * Get amount that can be refunded from a PayPal payment
     *
     * @param $payment
     * @return mixed
     */
    protected function getRefundableAmount($payment)
    {
        return collect($payment->transactions[0]->related_resources)
            ->sum(function ($resource) {
                if ($resource->sale) {
                    return $resource->sale->amount->total;
                }
                if ($resource->capture) {
                    return $resource->capture->amount->total;
                }
                return - $resource->refund->amount->total;
            });
    }
}
