<?php

namespace Mtc\Paypal;

use Carbon\Carbon;
use Illuminate\Support\Facades\App;
use Mtc\Basket\Contracts\BasketRepositoryInterface;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Paypal\Contracts\PayPalPaymentFactoryInterface;
use PayPal\Api\Amount;
use PayPal\Api\Patch;
use PayPal\Api\PatchRequest;
use PayPal\Api\Payer;
use PayPal\Api\Payment;
use PayPal\Api\PaymentExecution;
use PayPal\Api\RedirectUrls;
use PayPal\Api\Transaction;
use PayPal\Rest\ApiContext;

/**
 * Class PayPalPaymentFactory
 *
 * Class that handles payment flow for PayPal
 * Creates, updates and charges PayPal
 *
 * @package Mtc\Paypal
 */
class PayPalPaymentFactory implements PayPalPaymentFactoryInterface
{
    /**
     * Payment method constant
     *
     * @var string
     */
    const PAYMENT_METHOD = 'paypal';

    /**
     * Create paypal payment object from basket
     *
     * @param BasketRepositoryInterface $basket
     * @return Payment
     */
    public function create(BasketRepositoryInterface $basket): Payment
    {
        // Create new payer and method
        $payer = new Payer();
        $payer->setPaymentMethod(self::PAYMENT_METHOD);

        // Set redirect urls
        $redirect_urls = new RedirectUrls();
        $redirect_urls = $redirect_urls->setReturnUrl(route('paypal.api_return_url'))
            ->setCancelUrl(route('paypal.api_cancel_url'));

        // Set payment amount and currency
        $amount = new Amount();
        $amount = $amount->setCurrency($basket->getCurrency())
            ->setTotal($basket->getCostTotalAttribute());

        // Set transaction object
        $transaction = new Transaction();
        $transaction->setAmount($amount)
            ->setDescription(SITE_NAME . " Order");

        // Create the full payment object
        $payment = new Payment();
        $payment->setIntent(config('checkout.deferred_payments') ? 'authorize' : 'sale')
            ->setPayer($payer)
            ->setRedirectUrls($redirect_urls)
            ->setTransactions([
                $transaction
            ]);

        return $payment;
    }

    /**
     * Set payment reference
     *
     * @param string $reference
     * @param double $amount
     * @return Payment
     */
    public function createPaymentReference(string $reference, $amount)
    {
        \Mtc\Checkout\Invoice\Payment::query()
            ->create([
                'provider' => 'paypal',
                'amount' => $amount,
                'reference' => $reference
            ]);
    }

    /**
     * Update payment totals
     *
     * @param InvoiceRepositoryContract $invoice
     * @param Payment $payment
     * @return Payment|void
     */
    public function updateTotals(InvoiceRepositoryContract $invoice, Payment $payment)
    {
        $context = App::make(ApiContext::class);

        $patch = new Patch();
        $patch->setOp('replace')
            ->setPath('/transactions/0/amount')
            ->setValue([
                'currency' => $invoice->getCurrency(),
                'total' => number_format($invoice->getOutstandingAmount(), 2, '.', ''),
            ]);

        $patch_request = new PatchRequest();
        $patch_request->setPatches([
            $patch
        ]);

        $payment->update($patch_request, $context);
    }

    /**
     * Charge payment
     *
     * @param Payment $payment
     * @return array
     * @throws \Exception
     */
    public function chargePayment(Payment $payment, PaymentExecution $execution, ApiContext $context): array
    {
        $result = $payment->execute($execution, $context);
        $transactions = $result->getTransactions();
        $amount = $transactions[0]->getAmount();

        return [
            'provider' => 'paypal',
            'details' => $result->toArray(),
            'confirmed_at' => Carbon::now(),
            'amount' => $amount->getTotal() ?? 0,
        ];
    }
}
