<?php

namespace Mtc\Orders\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Event;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\InvoiceRepository;
use Mtc\Orders\Contracts\OrderAmendInvoiceFactoryContract;
use Mtc\Core\Gate\Auth;
use Mtc\Core\Node;
use Mtc\Foundation\Country;
use Mtc\Foundation\Http\Resources\CountryList;
use Mtc\Members\Contracts\MemberModel;
use Mtc\Orders\Contracts\OrderContract;
use Mtc\Orders\Events\FetchOrderProcessingActions;
use Mtc\Orders\Http\Resources\AdminOrderAjax;
use Mtc\Orders\Order;
use Mtc\Orders\OrderAmendState;
use Mtc\Orders\Status;

/**
 * Class OrderController
 *
 * @package Mtc\Orders
 */
class OrderController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @param Model $order
     * @return \Illuminate\Http\Response
     */
    public function index(Request $request, OrderContract $order)
    {
        /** @var Builder $query */
        $query = $order::query()
            ->with([
                'status',
                'billingAddress',
                'shippingAddress',
                'deliverySurcharge',
                'items'
            ])
            ->search($request)
            ->latest()
            ->paid();
        $this->page_meta['title'] = 'Manage Orders';
        return template('admin/orders/index.twig', [
            'orders' => $query->paginate(config('orders.admin.pagination'))->appends($request->input()),
            'page_meta' => $this->page_meta,
            'order_bulk_actions' => collect(\event('order_bulk_actions'))->filter()->values(),
            'extra_actions' => collect(\event('orders_index_extra_actions'))->filter()->values(),
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create(OrderContract $order)
    {
        $countries = Country::query()
            ->with('states')
            ->where('status', 1)
            ->get();

        $this->page_meta['title'] = 'Create Manual Order';
        return template('admin/orders/create.twig', [
            'order' => $order,
            'countries' => new CountryList($countries),
            'page_meta' => $this->page_meta,
        ]);
    }

    /**
     * Flush the basket to start a new one
     *
     * @param Request $request
     * @return \Illuminate\Http\RedirectResponse
     */
    public function flush(Request $request)
    {
        session()->forget('basket_id');
        return redirect()->back();
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        if ($request->has('member_id')) {
            $member = MemberModel::newQuery()->findOrNew($request->input('member_id', 0));
            if ($member->exists) {
                \Mtc\Modules\Members\Classes\Auth::login($member);
            }
        }


        App::make(config('orders.manual_order_basket_builder'))->handle();
        return redirect()->to(route('checkout.index'));
    }

    /**
     * Display the specified resource.
     *
     * @param int $id
     * @return \Illuminate\Http\Response|AdminOrderAjax
     */
    public function show(Request $request, $id, OrderContract $order)
    {
        $order = $order->find($id);
        if (!$order->exists) {
            session()->flash('error', 'Could not find order');
            return redirect()->to(route('orders.index'));
        }

        if ($request->wantsJson()) {
            return new AdminOrderAjax($order);
        }

        if (config('orders.assign_to_admin')) {
            $managers = app(config('auth.providers.users.model'))
                ->role(config('orders.assign_to_admin_role'))
                ->pluck('name', 'id');
        } else {
            $managers = false;
        }

        $this->page_meta['title'] = 'Order ' . $order->reference;
        return template('admin/orders/show.twig', [
            'order' => $order,
            'managers' => $managers,
            'page_meta' => $this->page_meta,
            'order_actions' => collect(Event::dispatch(new FetchOrderProcessingActions($order)))->filter()->values(),
            'order_status_actions' => Status::isNotTerminal($order->status_id)
                ? Status::getUpdateActionList($order->id)
                : [],
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id, OrderContract $order)
    {
        $countries = Country::query()
            ->with('states')
            ->where('status', 1)
            ->get();

        $order = $order->find($id);
        $this->page_meta['title'] = 'Edit Order ' . $order->reference;
        return template('admin/orders/edit.twig', [
            'order' => $order,
            'countries' => new CountryList($countries),
            'page_meta' => $this->page_meta,
            'order_statuses' => Status::query()->where('active', 1)->oldest('id')->get(),
        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param \Illuminate\Http\Request $request
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, Order $order, OrderAmendState $state)
    {

        $order->fill($request->input('details'));
        $state->from['details'] = $order->getOriginal();
        $state->to['details'] = $order->getDirty();

        if ($request->has('billing')) {
            $order->billingAddress->fill($request->input('billing'));
            $state->from['addresses']['billing'] = $order->billingAddress->getOriginal();
            $state->to['addresses']['billing'] = $order->billingAddress->getDirty();
            $order->billingAddress->save();
        }

        if ($request->has('shipping')) {
            $order->shippingAddress->fill($request->input('shipping'));
            $state->from['addresses']['shipping'] = $order->shippingAddress->getOriginal();
            $state->to['addresses']['shipping'] = $order->shippingAddress->getDirty();
            $order->shippingAddress->save();
        }

        if ($request->has('collection')) {
            $order->collectionAddress->fill($request->input('collection'));
            $state->from['addresses']['collection'] = $order->collectionAddress->getOriginal();
            $state->to['addresses']['collection'] = $order->collectionAddress->getDirty();
            $order->collectionAddress->save();
        }


        $request->has('surcharges') ? $this->amendSurcharges($request, $order, $state) : [];
        $request->has('discounts') ? $this->amendDiscounts($request, $order, $state) : [];

        $this->updateProductDetails($request, $order, $state);

        $order->vat_value = $order->cost_total->raw(true) - $order->cost_total->raw(false);
        $this->updateOrderTotals($request, $order);

        $order->save();

        if ($state->hasChanges()) {
            $order->history()
                ->create([
                    'triggered_by' => Auth::loadUser()->name,
                    'name' => 'Order updated',
                    'details' => "Values changed:\n " . $state->changeList()->implode("\n"),
                    'level' => 'warning',
                ]);
        }

        if (config('orders.create_invoice_when_editing_order')) {
            App::make(OrderAmendInvoiceFactoryContract::class)->create($order, $state);
        }

        return redirect()->to(route('orders.show', [$order->id]));
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param int $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        //
    }

    /**
     * Update Order Totals
     *
     * @param Request $request
     * @param Order $order
     */
    protected function updateOrderTotals(Request $request, Order $order)
    {
        if ($order->use_ex_vat) {
            // Set value as null to avoid confusion due to price calculations as the alternate is not used
            $order->fill([
                'cost_total' => $order->cost_total->raw(true),
                'cost_subtotal' => $order->cost_subtotal->raw(true),
            ]);
            return;
        }

        $order->fill([
            'cost_total_ex_vat' => $order->cost_total->raw(false),
            'cost_subtotal_ex_vat' => $order->cost_subtotal->raw(false),
        ]);
    }

    /**
     * Update Discount data
     *
     * @param Request $request
     * @param Order $order
     * @param OrderAmendState $state
     */
    protected function amendDiscounts(Request $request, Order $order, OrderAmendState $state)
    {
        $custom_discounts = collect($request->input('discounts'))
            ->filter(function ($discount) {
                return isset($discount['discount_type']) && $discount['discount_type'] === 'custom_discount';
            });

        if ($custom_discounts->count()) {
            $order->discounts()->delete();
            $custom_discounts->each(function ($discount) use ($order) {
                $order->discounts()
                    ->create([
                        'discount_type' => 'custom_discount',
                        'discount_id' => 0,
                        'discount_amount' => $discount['discount_amount'],
                        'discount_amount_ex_vat' => $discount['discount_amount_ex_vat'],
                        'name' => $discount['name']
                    ]);
            });
        }
    }

    /**
     * Update Surcharge data
     *
     * @param Request $request
     * @param Order $order
     * @return array
     */
    protected function amendSurcharges(Request $request, Order $order, OrderAmendState $state)
    {
        return collect($request->input('surcharges'))
            ->map(function ($surcharge_data, $id) use ($order, $state) {
                $surcharge = $order->surcharges()
                    ->firstOrNew([
                        'id' => $id
                    ]);
                if (!$surcharge) {
                    return null;
                }

                $surcharge->fill($surcharge_data);
                if ($surcharge->isDirty() === false) {
                    return null;
                }

                $state->from['discounts'][$id] = $surcharge->getOriginal();
                $state->to['discounts'][$id] = $surcharge->getDirty();

                // Set value as null to avoid confusion due to price calculations as the alternate is not used
                $surcharge->fill([
                    $order->use_ex_vat ? 'surcharge_amount' : 'surcharge_amount_ex_vat' => null,
                ]);

                $surcharge->save();
                return "Updated " . $surcharge->name;
            })
            ->filter()
            ->toArray();
    }

    /**
     * Update product details
     *
     * @param Request $request
     * @param Order $order
     * @return void
     */
    protected function updateProductDetails(Request $request, Order $order, OrderAmendState $state)
    {
        collect($request->input('products'))
            ->each(function ($product, $index) use ($order, $state) {
                // Find order item record
                $order_item = $order->items()
                    ->firstOrNew([
                        'id' => $product['order_item_id']
                    ]);

                // Delete item
                if ($product['delete'] === 'true') {
                    if ($order_item->exists) {
                        $state->from['items'][$order_item->id] = $order_item->getOriginal();
                        $state->to['items'][$order_item->id]['quantity'] = 0;
                        $order_item->delete();
                        return 'Removed product ' . $order_item->name;
                    }
                    return null;
                }


                // if we have product id specified
                if ($product['node_id']) {
                    $node = Node::query()->find($product['node_id']);
                    $purchasable = $node->nodeable;

                    $order_item->fill([
                        'purchasable_type' => get_class($purchasable),
                        'purchasable_id' => $purchasable->id,
                        'name' => $purchasable->name,
                        'sku' => $product['sku'],
                        'vat_rate' => $purchasable->getPrice()->taxRate(),
                    ]);
                }

                if (isset($product['quantity'])) {
                    $order_item->fill([
                        'attribute_fields' => json_decode($product['attribute_fields'], true),
                        'quantity' => $product['quantity'],
                    ]);
                }

                if (isset($product['percentage_discount'])) {
                    $order_item->fill([
                        'percentage_discount' => $product['percentage_discount']
                    ]);
                }

                if (isset($product['amount_discount'])) {
                    $order_item->fill([
                        'amount_discount' => $product['amount_discount']
                    ]);
                }

                $product_price_column = $order->use_ex_vat ? 'paid_price_ex_vat' : 'paid_price';
                if (isset($product[$product_price_column])) {
                    $order_item->amendPrice($product[$product_price_column], $order->use_ex_vat);
                }

                if ($order_item->isDirty() === false) {
                    return null;
                }

                $state->from['items'][$order_item->id] = $order_item->getOriginal();
                $state->to['items'][$order_item->id] = $order_item->getDirty();
                $order_item->save();

                if ($order_item->wasRecentlyCreated) {
                    $state->from['items'][$order_item->id] = $state->from['items'][''];
                    $state->to['items'][$order_item->id] = $state->to['items'][''];
                    unset($state->from['items']['']);
                    unset($state->to['items']['']);
                }
            });
    }

    /**
     * Display a listing of the resource.
     *
     * @param Model $order
     * @return \Illuminate\Http\Response
     */
    public function unpaid(Request $request, OrderContract $order)
    {
        /** @var Builder $query */
        $query = $order::query()
            ->whereNull('paid_at')
            ->search($request)
            ->latest();
        $this->page_meta['title'] = 'View Unpaid Orders';
        return template('admin/orders/unpaid.twig', [
            'orders' => $query->paginate(config('orders.admin.pagination')),
            'page_meta' => $this->page_meta,
        ]);
    }

    /**
     * Mark order as paid manually
     *
     * @param $order_id
     * @param OrderContract $order
     * @return \Illuminate\Http\RedirectResponse
     */
    public function markPaid($order_id, OrderContract $order)
    {
        $order = $order->find($order_id);
        if ($order === null || !empty($order->paid_at)) {
            session()->flash('error', 'Failed to process order');
            return back();
        }

        $order->invoices
            ->each(function ($invoice) {
                /** @var InvoiceRepository $repository */
                $repository = App::make(InvoiceRepositoryContract::class);
                $repository->setModel($invoice);
                $repository->savePayment([
                    'provider' => 'manual-order',
                    'amount' => $repository->getOutstandingAmount(),
                    'reference' => config('orders.manual_order_reference_prefix') . $repository->getId(),
                    'details' => [],
                    'confirmed_at' => now(),
                ]);
            });

        $order->history()
            ->create([
                'triggered_by' => Auth::loadUser()->name,
                'name' => 'Marked as Paid by Admin',
                'level' => 'success',
            ]);

        session()->flash('success', 'Order processed');
        return back();
    }

    /**
     * Change the person who manages this invoice
     *
     * @param Request $request
     * @param int $id
     * @param OrderContract $order
     * @return \Illuminate\Http\RedirectResponse
     */
    public function assignAdmin(Request $request, $id, OrderContract $order)
    {
        $order = $order->find($id);
        $order->managing_user_id = $request->input('user_id');
        $order->save();

        if ($request->wantsJson()) {
            return response('success');
        }

        session()->flash('success', 'Manager updated');
        return back();
    }

    /**
     * Set tracking details on order
     *
     * @param Request $request
     * @param $order_id
     * @param OrderContract $order
     * @return \Illuminate\Http\RedirectResponse
     */
    public function setTracking(Request $request, $order_id, OrderContract $order)
    {
        $order = $order->find($order_id);
        $order->tracking_data = collect($request->input())->forget('action');
        $order->save();

        if ($order->wasChanged('tracking_data')) {
            $order->history()
                ->create([
                    'triggered_by' => Auth::loadUser()->name,
                    'name' => 'Set Tracking details',
                    'level' => 'success',
                    'details' => collect($order->tracking_data)
                        ->map(function ($value, $key) {
                            return "{$key}: {$value}";
                        })->implode("\n"),
                ]);
        }

        session()->flash('success', 'Tracking Details saved');
        if ($request->input('action') === 'open_shipment_creation') {
            return redirect(route('shipment.create', [ $order->id ]));
        }
        return back();
    }
}
