<?php

namespace Mtc\Orders;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Http\Request;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Event;
use Mtc\Checkout\Contracts\LinkableToAdmin;
use Mtc\Checkout\Invoice;
use Mtc\Core\Currency;
use Mtc\Members\Member;
use Mtc\Money\HasPrices;
use Mtc\Orders\Contracts\OrderContract;
use Mtc\Orders\Events\OrderPaid;
use Mtc\Orders\Events\OrderStatusChange;
use Mtc\Returns\ReturnRequest;

/**
 * Class Order
 *
 * @package Mtc\Orders
 */
class Order extends Model implements OrderContract, LinkableToAdmin
{
    use HasPrices;

    /**
     * Mass assignable attributes
     *
     * @var array
     */
    protected $fillable = [
        'reference',
        'status_id',
        'basket_id',
        'invoice_id',
        'member_id',
        'cost_subtotal',
        'cost_subtotal_ex_vat',
        'cost_total',
        'cost_total_ex_vat',
        'backorder_cost_total',
        'vat_value',
        'use_ex_vat',
        'currency_code',
        'type',
        'email',
        'message',
        'customer_name',
        'contact_number',
        'secondary_phone_number',
        'newsletter_signup',
        'additional_info',
        'tracking_data',
    ];

    /**
     * Cast variables to specific types
     *
     * @var array
     */
    protected $casts = [
        'paid_at' => 'datetime',
        'newsletter_signup' => 'boolean',
        'additional_info' => 'array',
        'tracking_data' => 'array',
        'created_at' => 'datetime',
        'updated_at' => 'datetime',
    ];

    /**
     * Fields that convert to price
     *
     * @var array
     */
    protected $price_fields = [
        'cost_subtotal_ex_vat' => 'cost_subtotal',
        'cost_total_ex_vat' => 'cost_total',
    ];

    /**
     * Column that sets order tax rate
     *
     * @var string
     */
    protected $tax_attribute = 'vat_rate';

    /**
     * Extend model booting
     */
    protected static function boot()
    {
        parent::boot();

        // When deleting an order clean out its related tables
        self::deleting(function (self $order) {
            $order->removeRelations();
        });

        self::retrieved(function (self $order) {
            if ($order->attributes['use_ex_vat']) {
                $order->load_ex_vat_price = true;
            }
        });
    }

    /**
     * Relationship with invoices for this order
     *
     * @return \Illuminate\Database\Eloquent\Relations\MorphMany
     */
    public function invoices()
    {
        return $this->morphMany(Invoice::class, 'payable');
    }

    /**
     * Relationship with member who owns this basket
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function member()
    {
        return $this->belongsTo(Member::class, 'member_id');
    }

    /**
     * Relationship with basket from which this was created
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function basket()
    {
        return $this->belongsTo(config('basket.model'), 'basket_id');
    }

    /**
     * Relationship with addresses
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function addresses()
    {
        return $this->hasMany(Address::class);
    }

    /**
     * Shipping address relationship
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany|\Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function shippingAddress()
    {
        return $this->hasOne(Address::class)
            ->where('type', Address::TYPE_SHIPPING);
    }

    /**
     * Shipping address relationship
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany|\Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function collectionAddress()
    {
        return $this->hasOne(Address::class)
            ->where('type', Address::TYPE_COLLECTION);
    }

    /**
     * Shipping address relationship
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany|\Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function billingAddress()
    {
        return $this->hasOne(Address::class)
            ->where('type', Address::TYPE_BILLING);
    }

    /**
     * Relationship with all discounts
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function discounts()
    {
        return $this->hasMany(Discount::class);
    }

    /**
     * Relationship with all surcharges
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function surcharges()
    {
        return $this->hasMany(Surcharge::class);
    }

    /**
     * Relationship with all return requests for this order
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function returnRequests()
    {
        if (config('returns.enabled')) {
            return $this->hasMany(ReturnRequest::class);
        }
    }

    /**
     * Relationship with delivery surcharge
     * @return \Illuminate\Database\Eloquent\Relations\HasMany|\Illuminate\Database\Eloquent\Relations\HasOne
     */
    public function deliverySurcharge()
    {
        return $this->hasOne(Surcharge::class)
            ->where('surcharge_type', 'delivery');
    }

    /**
     * Relationship with order items
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function items()
    {
        return $this->hasMany(Item::class);
    }

    /**
     * Relationship with order status
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo|\Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function status()
    {
        return $this->belongsTo(Status::class);
    }

    /**
     * Relationship with order manager
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo|\Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function managingUser()
    {
        return $this->belongsTo(config('auth.providers.users.model'));
    }

    /**
     * Relationship with historic details of the order
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function history()
    {
        return $this->hasMany(History::class)
            ->orderBy('created_at', 'desc');
    }

    /**
     * Relationship with shipments this order has
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function shipments()
    {
        return $this->hasMany(OrderShipment::class);
    }

    /**
     * Relationship with Integration history
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function integrationHistory()
    {
        return $this->hasMany(IntegrationHistory::class);
    }

    /**
     * Relationship with currency order was created
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function currency()
    {
        return $this->belongsTo(Currency::class, 'currency_code', 'currency');
    }

    /**
     * Next Paid order
     *
     * @return mixed
     */
    public function next()
    {
        return self::query()
            ->paid()
            ->where('id', '>', $this->id)
            ->orderBy('id')
            ->firstOrNew([]);
    }

    /**
     * Previous paid order
     *
     * @return mixed
     */
    public function previous()
    {
        return self::query()
            ->paid()
            ->where('id', '<', $this->id)
            ->orderByDesc('id')
            ->firstOrNew([]);
    }

    /**
     * Relationship with other orders from this user
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function pastOrders()
    {
        return $this->hasMany(static::class, 'member_id', 'member_id')
            ->whereNotNull('member_id')
            ->where('id', '!=', $this->id)
            ->latest();
    }

    /**
     * Relationship with order from this basket that has been paid
     *
     * @return \Illuminate\Database\Eloquent\Relations\HasMany
     */
    public function paidSibling()
    {
        return $this->hasOne(static::class, 'basket_id', 'basket_id')
            ->paid();
    }

    /**
     * ->payment_method to find what method is used to pay for order
     * String by default but supports array mode
     *
     * @param bool $as_string
     * @return Collection|string
     */
    public function getPaymentMethodAttribute($as_string = true)
    {
        $methods = $this->invoices
            ->map(function ($invoice) {
                return $invoice->payments->map(function ($payment) {
                    if ($payment->details['payment_method_type']) {
                        return $payment->provider . ' (' . $payment->details['payment_method_type'] . ')';
                    }
                    return $payment->provider;
                });
            })
            ->flatten(1)
            ->unique();

        if ($as_string === false) {
            return $methods;
        }

        return $methods
            ->map(function ($gateway) {
                return ucwords(str_replace('-', ' ', $gateway));
            })
            ->implode(',');
    }

    /**
     * Filter only paid orders
     *
     * @param Builder $query
     * @return Builder|\Illuminate\Database\Query\Builder
     */
    public function scopePaid(Builder $query)
    {
        return $query->whereNotNull('paid_at');
    }

    /**
     * Add relationships
     *
     * @param Builder $query
     * @return Builder|\Illuminate\Database\Query\Builder
     */
    public function scopeWithDetails(Builder $query)
    {
        return $query->with([
            'items.lines',
            'discounts',
            'surcharges',
            'history',
        ]);
    }

    /**
     * @param Request $request
     * @return Builder
     */
    public function scopeSearch(Builder $query, Request $request)
    {
        collect(config('orders.admin_search_filters', []))
            ->each(function($filter_class) use ($request, $query) {
                App::make($filter_class)->handle($request, $query);
            });

        return $query;
    }

    /**
     * Mark order as Paid
     */
    public function markPaid($reference, $custom_status = null): void
    {
        // Don't allow to mark order as paid twice
        if (!empty($this->paid_at)) {
            return;
        }

        $this->reference = $reference;
        $this->paid_at = Carbon::now();
        $this->status_id = $custom_status ?: $this->getOrderConfirmationStatus();
        $this->save();
        Event::dispatch(new OrderPaid($this));
    }

    /**
     * Method for defining default status when order gets confirmed
     *
     * @return \Illuminate\Config\Repository|\Illuminate\Contracts\Foundation\Application|mixed
     */
    protected function getOrderConfirmationStatus()
    {
        return config('orders.statuses.confirmed');
    }

    /**
     * Check if order has delivery
     *
     * @return bool
     */
    public function hasDelivery(): bool
    {
        return $this->deliverySurcharge !== null;
    }

    /**
     * Remove unpaid orders from this basket
     */
    public function removeUnpaidBasketOrders()
    {
        self::query()
            ->where('basket_id', $this->basket_id)
            ->whereNull('paid_at')
            ->where('id', '!=', $this->id)
            ->get()
            ->each(function (self $unpaid_order) {
                $unpaid_order->delete();
            });
    }

    /**
     * Clean out relationships when deleting a record
     */
    protected function removeRelations()
    {
        $this->addresses()->delete();
        $this->surcharges()->delete();
        $this->discounts()->delete();
        $this->invoices
            ->each(function ($invoice) {
                $invoice->delete();
            });
        $this->items
            ->each(function ($item) {
                $item->lines()->delete();
                $item->forceDelete();
            });
    }

    /**
     * Change status on order
     *
     * @param $new_status_id
     * @param $admin_user
     */
    public function changeStatus($new_status_id, $admin_user)
    {
        $this->status_id = $new_status_id;
        if ($this->isDirty(['status_id']) == false) {
            return;
        }

        $this->save();
        Event::dispatch(new OrderStatusChange($this, $this->status_id));
        session()->flash('success', 'Order Status Updated');

        $this->load('status');
        $this->history()
            ->create([
                'level' => 'success',
                'name' => 'Changed Status to ' . $this->status->name,
                'triggered_by' => $admin_user->name ?? '',
            ]);

    }

    /**
     * Get the name of this payable
     *
     * @return mixed
     */
    public function getName()
    {
        return 'Order';
    }

    /**
     * Get the link to this item in admin
     *
     * @return mixed
     */
    public function getAdminLink()
    {
        return route('orders.show', [ $this->id ]);
    }

    /**
     * Check if order is fully shipped
     *
     * @return bool
     */
    public function isFullyShipped()
    {
        return $this->items
            ->reject(function ($order_item) {
                return $order_item->isShipped();
            })
            ->count() == 0;
    }

    /**
     * Check if order is fully shipped
     *
     * @return bool
     */
    public function hasItemsWithoutShipment()
    {
        return $this->items
            ->filter(function ($order_item) {
                return $order_item->quantity > $order_item->shipmentItems()->sum('quantity');
            })
            ->count() == 0;
    }

    /**
     * Check if it is possible to repeat this order.
     * This is possible if at least one item of the order is still available
     *
     * @return bool
     */
    public function canReorder()
    {
        return $this->items
                ->filter(function ($item){
                    return $item->purchasable;
                })
                ->count() > 0;

    }

    /**
     * For order the tax value is already calculated in vat_value we just need to determine
     * whether order has or has not vat
     *
     * @param $key
     * @return float|int
     */
    public function taxAttribute($key)
    {
        if ((double)$this->attributes['vat_value'] >= 0.01) {
            return config('tax.vat_rates.' . config('tax.default_vat_rate'));
        }

        return 0;
    }
}
