<?php

namespace Mtc\Orders\Listeners;

use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Events\PaidInvoice;
use Mtc\Orders\Order;

/**
 * Class MarkOrderPaidFromInvoice
 *
 * @package Mtc\Orders
 */
class MarkOrderPaidFromInvoice
{

    /**
     * Handle the event.
     *
     * @param PaidInvoice $event
     * @return void
     */
    public function handle(PaidInvoice $event)
    {
        if (!$this->isInvoiceForOrder($event->invoice)) {
            return;
        }

        /** @var Order $order */
        $order = $event->invoice->getPayable();
        $confirmation_status = $event->payment_details['confirmation_status'] ?? null;
        $order->markPaid($this->getReference($event, $order), $confirmation_status);

        $order->history()
            ->create([
                'level' => 'success',
                'name' => 'Order confirmed with Payment reference ' . $event->reference,
                'triggered_by' => 'System'
            ]);
        $order->removeUnpaidBasketOrders();
    }

    /**
     * Check if invoice is for an order payment
     *
     * @return bool
     */
    protected function isInvoiceForOrder(InvoiceRepositoryContract $invoice)
    {
        $record = $invoice->getPayable();
        return $record && get_class($record) === config('orders.order_class');
    }

    /**
     * Get reference that should be set on orders
     *
     * @param PaidInvoice $event
     * @param $order
     * @return string
     */
    protected function getReference(PaidInvoice $event, $order)
    {
        if (config('orders.use_payment_reference_for_order_reference')) {
            return $event->reference;
        }

        return config('orders.order_reference_prefix', '') . $order->id;
    }
}
