<?php

namespace Mtc\Orders;

use Illuminate\Database\Eloquent\Model;
use Mtc\Money\HasPrices;
use Mtc\Money\Price;

/**
 * Class Discount
 *
 * @package Mtc\Orders
 */
class Discount extends Model
{
    use HasPrices;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'order_discounts';

    /**
     * Columns that cannot be mass assigned
     *
     * @var array
     */
    protected $guarded = [
        'id'
    ];

    /**
     * When saved touch an update of a relationship
     *
     * @var array
     */
    protected $touches = [
        'order'
    ];


    /**
     * Boot model
     */
    protected static function boot()
    {
        parent::boot();

        self::retrieved(function (self $order_item) {
            if ($order_item->order->use_ex_vat) {
                $order_item->load_ex_vat_price = true;
            }
        });
    }

    /**
     * Relationship with Basket
     *
     * @return \Illuminate\Database\Eloquent\Relations\BelongsTo
     */
    public function order()
    {
        return $this->belongsTo(config('orders.order_class'));
    }

    /**
     * Relationship with discount
     *
     * @return \Illuminate\Database\Eloquent\Relations\MorphTo
     */
    public function discount()
    {
        return $this->morphTo('discount');
    }

    /**
     * discount_amount as Price object
     *
     * @return Price
     */
    public function getDiscountAmountAttribute(): Price
    {
        if ($this->load_ex_vat_price) {
            $config = [
                'price_entered_with_tax' => false
            ];
            $price = new Price($this->attributes['discount_amount_ex_vat'], $this->order->vat_rate, $config);
        } else {
            $price = new Price($this->attributes['discount_amount'], $this->order->vat_rate);
        }
        $price->calculate();
        return $price;
    }
}
