<?php

namespace Mtc\Orders;

use \Illuminate\Database\Eloquent\Model as EloquentModel;
use \Illuminate\Database\Eloquent\Builder;

/**
 * Order\OrderStatus class. This looks after the order_statuses table
 * using Eloquent base modelling
 *
 * @author Davis Lasis <davis.lasis@mtcmedia.co.uk>
 * @package Mtc\Orders
 */
class OrderStatus extends EloquentModel
{
    /**
     * Constants for Order Statuses (status_id)
     */
    const ORDER_RECEIVED = 0;
    const PAYMENT_ACCEPTED = 1;
    const PAYMENT_REJECTED = 2;
    const ON_PICKING = 3;
    const SHIPPED = 4;
    const PART_REFUNDED = 5;
    const REFUNDED = 6;
    const PART_SHIPPED = 7;
    const WAITING_FOR_RESPONSE = 8;
    const CANCELLED = 9;

    /**
     * @var string $table The table name to use with the model
     */
    protected $table = 'order_statuses';

    /**
     * The attributes that are mass assignable.
     *
     * @var array $fillable
     */
    protected $fillable = [
        'status_id',
        'name',
        'send_email',
        'email_path',
        'email_text',
        'is_failure',
        'active'
    ];

    /**
     * Indicates if the model should be timestamped.
     *
     * @var bool
     */
    public $timestamps = true;

    /**
     * Scope - active()
     *
     * @param Builder $query Query to extend
     * @return Builder
     */
    public function scopeActive(Builder $query)
    {
        return $query->where('active', 1);
    }

    /**
     * Scope - isFailure()
     *
     * @param Builder $query Query to extend
     * @return Builder
     */
    public function scopeIsFailure(Builder $query)
    {
        return $query->where('is_failure', 1);
    }

    /**
     * Scope - notFailure()
     *
     * @param Builder $query Query to extend
     * @return Builder
     */
    public function scopeNotFailure(Builder $query)
    {
        return $query->where('is_failure', 0);
    }

    /**
     * Get Order Status by status_id and return name.
     *
     * @return string
     */
    public static function getNameByStatusId($status_id)
    {
        return self::query()->where('status_id', $status_id)->first()->name ?? '';
    }

    /**
     * Get list of all Order Statuses which has email_path defined.
     *
     * @return array
     */
    public static function getSendEmailList()
    {
        $output = [];
        $order_statuses = self::where('send_email', 1)->active()->get();
        foreach ($order_statuses as $status) {
            $output[$status->status_id] = $status->email_path;
        }
        return $output;
    }

}
