<?php

namespace Mtc\Orders\Factories;

use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Event;
use Mtc\Basket\Basket;
use Mtc\Basket\Contracts\BasketRepositoryInterface;
use Mtc\Basket\Http\Resources\CheckoutAjax;
use Mtc\Checkout\Contracts\PayableContract;
use Mtc\Orders\Contracts\OrderFactoryContract;
use Mtc\Orders\Contracts\OrderRepositoryContract;
use Mtc\Orders\Events\OrderCreated;
use Mtc\Orders\Order;

/**
 * Class OrderFactory
 *
 * @package Mtc\Orders
 */
class OrderFactory implements OrderFactoryContract
{
    /**
     * Create an order from basket
     *
     * @param BasketRepositoryInterface $basket
     * @return PayableContract
     */
    public function create(BasketRepositoryInterface $basket_repository): PayableContract
    {
        // Make sure any events triggered in validation process are applied during order creation as well
        // E.g. if validation of a field changes VAT on basket or pricing
        $is_valid_basket = collect(config('basket.checkout_validation', []))
            ->reject(function ($validator_class) {
                return App::make($validator_class)->handle();
            })->isEmpty();


        /** @var Basket $basket */
        $basket = $basket_repository->getModel();
        /** @var Order $order */
        $order = Order::query()
            ->create([
                'status_id' => 0,
                'basket_id' => $basket->id,
                'member_id' => $basket->member_id,
                'cost_subtotal' => $basket->subtotal,
                'cost_subtotal_ex_vat' => $basket->subtotal_ex_vat,
                'cost_total' => $basket->cost_total,
                'cost_total_ex_vat' => $basket->cost_total_ex_vat,
                'use_ex_vat' => $basket_repository->shouldUseExVat(),
                'backorder_cost_total' => $basket->backorder_cost_total,
                'email' => $basket->email,
                'contact_number' => $basket->contact_number,
                'message' => $basket->message,
                'currency_code' => session('selected_currency'),
                'secondary_phone_number' => $basket->secondary_phone_number,
                'newsletter_signup' => $basket->newsletter_signup,
                'additional_info' => $basket->additional_info,
                'tracking_data' => $basket->tracking_data,
                'customer_name' => $basket_repository->getCustomerName(),
                'vat_value' => $basket_repository->getVatValue(),
                'type' => $basket_repository->getType(),
            ]);

        $basket->discounts
            ->each(function ($basket_discount) use ($order) {
                $order->discounts()
                    ->create([
                        'discount_id' => $basket_discount->discount_id,
                        'discount_type' => $basket_discount->discount_type,
                        'discount_amount' => $basket_discount->getPriceAttribute(true),
                        'name' => $basket_discount->getNameAttribute(),
                    ]);
            });

        $order->addresses()->create($basket->billingAddress->toArray());

        if ($basket->shipping_type === 'delivery') {
            $order->addresses()->create($basket->shippingAddress->toArray());
        } else {
            $this->setOrderCollectionAddress($basket, $order);
        }

        $basket->surcharges
            ->each(function ($basket_surcharge) use ($order) {
                $order->surcharges()
                    ->create([
                        'surcharge_id' => $basket_surcharge->surcharge_id,
                        'surcharge_type' => $basket_surcharge->surcharge_type,
                        'surcharge_amount' => $basket_surcharge->getPriceAttribute(true),
                        'name' => $basket_surcharge->getNameAttribute(),
                    ]);
            });

        $basket->items
            ->each(function ($basket_item) use ($order) {
                $discounted_values = $basket_item->getDiscountedValues();
                $order_item = collect([
                    'purchasable_id' => $basket_item->purchasable_id,
                    'purchasable_type' => $basket_item->purchasable_type,
                    'quantity' => $basket_item->quantity,
                    'quantity_on_backorder' => $basket_item->backorder_quantity,
                    'name' => $basket_item->purchasable->getName(),
                    'sku' => $basket_item->purchasable->getSku(),
                    'attribute_fields' => $basket_item->attribute_fields,
                    'data' => $basket_item->data,
                    'unit_price' => $basket_item->getOriginalPriceAttribute(true),
                    'unit_price_ex_vat' => $basket_item->getOriginalPriceAttribute(false),
                    'paid_price' => $basket_item->getUnitPriceAttribute(true),
                    'paid_price_ex_vat' => $basket_item->getUnitPriceAttribute(false),
                    'vat_rate' => $basket_item->tax_rate,
                ])
                    ->merge($discounted_values)
                    ->toArray();

                $order->items()->create($order_item);
            });

        if (!empty($basket->message)) {
            $order->history()
                ->create([
                    'level' => 'success',
                    'name' => 'Customer Added Note',
                    'details' => $basket->message,
                    'triggered_by' => $basket_repository->getCustomerName()
                ]);
        }

        Event::dispatch(new OrderCreated($basket, $order));

        /** @var OrderRepositoryContract $payable */
        $payable = App::make(OrderRepositoryContract::class);
        $payable->setModel($order);
        return $payable;
    }

    /**
     * Set the collection address for order
     *
     * @param Basket $basket
     * @param Order $order
     */
    protected function setOrderCollectionAddress($basket, $order)
    {
        $collect_location = $basket->surcharges
            ->where('surcharge_type', 'collection')
            ->first()
            ->surcharge
            ->only(\Mtc\Foundation\Address::$address_fields);

        $collect_location['type'] = 'collection';
        $collect_location['first_name'] = $basket->billingAddress->first_name;
        $collect_location['last_name'] = $basket->billingAddress->last_name;
        $order->addresses()
            ->create($collect_location);
    }
}
