<?php

namespace Mtc\Orders;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;

/**
 * Class Status
 *
 * @package Mtc\Orders
 */
class Status extends Model
{
    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'order_statuses';

    /**
     * Mass assignable attributes
     *
     * @var array
     */
    protected $fillable = [
        'name',
        'email_subject',
        'email_text',
        'email_template_path',
        'send_email_on_status',
        'is_failure_status',
        'active',
        'colour_code',
    ];

    /**
     * Cast attributes to specific types
     *
     * @var array
     */
    protected $casts = [
        'active' => 'boolean',
        'send_email_on_status' => 'boolean',
        'is_failure_status' => 'boolean',
    ];

    /**
     * Get the list for status update urls on available statuses
     *
     * @param $order_id
     * @return Collection
     */
    public static function getUpdateActionList($order_id)
    {
        return self::query()
            ->where('active', 1)
            ->oldest('id')
            ->get()
            ->map(function(self $status) use ($order_id) {
                return [
                    'name' => $status->name,
                    'method' => 'post',
                    'action' => route('order.set_status', [
                        'id' => $order_id,
                        'status_id' => $status->id,
                    ])
                ];
            });
    }

    /**
     * Check if the current status is not in terminal status list
     *
     * @param $current_status_id
     * @return bool
     */
    public static function isNotTerminal($current_status_id)
    {
        return collect(config('orders.terminal_statuses'))
            ->map(function ($status) use ($current_status_id)  {
                return config("orders.statuses.{$status}") == $current_status_id;
            })
            ->filter()
            ->isEmpty();
    }

    /**
     * Get the email subject for status update
     *
     * @param Order $order
     * @return string
     */
    public function getEmailSubject($order)
    {
        return $this->replaceContent($order, $this->email_subject);
    }

    /**
     * Get the status text for order
     *
     * @param Order $order
     * @return string
     */
    public function getStatusText($order)
    {
        return $this->replaceContent($order, $this->email_text);
    }

    /**
     * Replace content for email
     *
     * @param Order $order
     * @param string $field
     * @return string|string[]
     */
    protected function replaceContent($order, $field)
    {
        $replacements = config('orders.status_replacements', []);
        $replace_values = collect($replacements)
            ->keyBy(function ($field, $tag) {
                return "{{{$tag}}}";
            })
            ->map(function ($field) use ($order) {
                try {
                    return $this->mapReplacement($field, $order);
                } catch (\Exception $exception) {
                    return '';
                }
            });

        return str_replace(
            $replace_values->keys()->toArray(),
            $replace_values->values()->toArray(),
            $field
        );
    }

    /**
     * Map the value for a field on model
     * @param $field
     * @param $model
     * @return \Illuminate\Config\Repository|mixed
     * @throws \Exception
     */
    protected function mapReplacement($field, $model)
    {
        if (config($field) !== null) {
            return config($field);
        }

        $field_parts = explode('.', $field);
        while(count($field_parts) > 1) {
            $relationship_name = array_shift($field_parts);
            $model = $model->{$relationship_name};
        }

        $field_name = array_shift($field_parts);
        if ($model->{$field_name} instanceof Carbon) {
            return $model->{$field_name}->format(config('orders.status_replacement_date_format'));
        }

        return $model[$field_name];
    }
}
