<?php

namespace Mtc\Notifications\Tests\Unit;

use Illuminate\Foundation\Testing\DatabaseMigrations;
use Illuminate\Foundation\Testing\RefreshDatabase;
use Illuminate\Foundation\Testing\WithFaker;
use Mtc\Notifications\Contracts\NotificationModel;
use Mtc\Notifications\Facades\Notification;
use Mtc\Notifications\Tests\TestCase;
use Mtc\Notifications\Tests\User;

class NotificationRepositoryTest extends TestCase
{
    use DatabaseMigrations;

    public function test_adding_notification(): void
    {
        $notification = Notification::addNotification('info', 'test', 'details', 'lorem', 'source');

        $exists = $this->model()->newQuery()
            ->where('severity', 'info')
            ->where('message', 'test')
            ->exists();

        self::assertInstanceOf(NotificationModel::class, $notification);
        self::assertTrue($exists);
    }

    public function test_has_unread_no_messages(): void
    {
        self::assertFalse(Notification::hasUnread($this->testUser()));
    }

    public function test_has_unread_has_visible_messages(): void
    {
        Notification::addNotification('info', 'test', 'details', 'lorem', 'source');
        self::assertTrue(Notification::hasUnread($this->testUser()));
    }

    public function test_has_unread_has_invisible_messages(): void
    {
        Notification::addNotification('info', 'test', 'details', 'custom-permission', 'source');
        self::assertFalse(Notification::hasUnread($this->testUser()));
    }

    public function test_has_unread_has_mixed_visibility_messages(): void
    {
        Notification::addNotification('info', 'test', 'details', 'lorem', 'source');
        Notification::addNotification('info', 'test', 'details', 'custom-permission', 'source');
        self::assertTrue(Notification::hasUnread($this->testUser()));
    }

    public function test_get_all(): void
    {
        Notification::addNotification('info', 'test', 'details', 'lorem', 'source');
        Notification::addNotification('info', 'test', 'details', 'custom-permission', 'source');
        Notification::addNotification('warning', 'test2', 'details', 'ipsum', 'source');
        $notifications = Notification::getAll($this->testUser());
        self::assertEquals(2, $notifications->count());
    }

    public function test_mark_read(): void
    {
        $notification = Notification::addNotification('info', 'test', 'details', 'lorem', 'source');
        Notification::markRead($notification, $this->testUser());

        $model = $this->model()->newQuery()->findOrFail($notification->id);
        self::assertNotNull($model->read_at);
    }

    public function test_mark_unread(): void
    {
        $notification = Notification::addNotification('info', 'test', 'details', 'lorem', 'source');
        Notification::markRead($notification, $this->testUser());
        Notification::markUnread($notification, $this->testUser());

        $model = $this->model()->newQuery()->findOrFail($notification->id);
        self::assertNull($model->read_at);
    }

    public function test_destroy(): void
    {
        $notification = Notification::addNotification('info', 'test', 'details', 'lorem', 'source');
        Notification::destroy($notification);

        $exists = $this->model()->newQuery()
            ->where('id', $notification->id)
            ->exists();

        self::assertFalse($exists);

    }

    private function model(): NotificationModel
    {
        return app(NotificationModel::class);
    }

    private function testUser(): User
    {
        return new User([ 'id' => 1 ]);
    }
}
