<?php

namespace Mtc\Notifications;

use Carbon\Carbon;
use Illuminate\Contracts\Auth\Authenticatable;
use Illuminate\Http\Resources\Json\JsonResource;
use Illuminate\Support\Facades\DB;
use Mtc\Notifications\Contracts\NotificationModel;
use Mtc\Notifications\Http\Resources\NotificationList;

class NotificationRepository
{
    public function __construct(protected NotificationModel $notification_model)
    {
        //
    }

    public function addNotification(
        string $severity,
        string $message,
        $details,
        $required_permission,
        $source
    ): NotificationModel {
        return $this->notification_model->newQuery()
            ->create([
                'severity' => $severity,
                'message' => $message,
                'details' => is_string($details) ? $details : json_encode($details),
                'source' => $source,
                'recipient_permission_level' => $required_permission,
            ]);
    }

    /**
     * Check if user has some unread notifications
     *
     * @param Authenticatable $user
     * @return bool
     */
    public function hasUnread(Authenticatable $user): bool
    {
        return $this->notification_model
            ->newQuery()
            ->whereNull('read_at')
            ->where(function ($permission_check_query) use ($user) {
                $permission_check_query->whereNull('recipient_permission_level')
                    ->orWhereIn('recipient_permission_level', $user->getAllPermissions()->pluck('name'));
            })
            ->exists();
    }

    /**
     * Get all notifications, including read
     *
     * @param Authenticatable $user
     * @param int|null $per_page
     * @param int $page
     * @return JsonResource
     */
    public function getAll(Authenticatable $user, $per_page = null, $page = 1)
    {
        $perPage = $per_page ?? $this->notification_model->getPerPage();

        $notifications = $this->notification_model
            ->newQuery()
            ->where(function ($permission_check_query) use ($user) {
                $permission_check_query->whereNull('recipient_permission_level')
                    ->orWhereIn('recipient_permission_level', $user->getAllPermissions()->pluck('name'));
            })
            ->orderBy('read_at', 'asc')
            ->orderBy('created_at', 'desc')
            ->paginate($perPage, ['*'], 'page', $page);

        return new NotificationList($notifications);
    }

    /**
     * Mark one or more notifications as read
     *
     * @param $notification_ids
     * @param Authenticatable $user
     * @return bool
     */
    public function markRead($notification_ids, Authenticatable $user): bool
    {
        if (is_numeric($notification_ids)) {
            $notification_ids = [ $notification_ids ];
        }

        return $this->notification_model
            ->newQuery()
            ->whereIn('id', $notification_ids)
            ->update([
                'read_at' => Carbon::now(),
                'read_by' => $user->id,
            ]);
    }

    /**
     * Mark one or more notifications as unread
     *
     * @param $notification_ids
     * @return bool
     */
    public function markUnread($notification_ids): bool
    {
        if (is_numeric($notification_ids)) {
            $notification_ids = [ $notification_ids ];
        }

        return $this->notification_model
            ->newQuery()
            ->whereIn('id', $notification_ids)
            ->update([
                'read_at' => null,
                'read_by' => null,
            ]);
    }

    /**
     * Remove notifications
     *
     * @param $notification_ids array
     * @return mixed
     */
    public function destroy($notification_ids): bool
    {
        if (is_numeric($notification_ids)) {
            $notification_ids = [ $notification_ids ];
        }
        return $this->notification_model
            ->newQuery()
            ->whereIn('id', $notification_ids)
            ->delete();
    }
}
