<?php

namespace Mtc\MultiBuy\Providers;

use Illuminate\Database\Eloquent\Relations\Relation;
use Illuminate\Support\ServiceProvider;
use Mtc\Basket\Discount;
use Mtc\MultiBuy\Console\Commands\InstallDiscounts;
use Mtc\MultiBuy\Contracts\BasketDiscountRepositoryInterface;
use Mtc\MultiBuy\Contracts\DiscountRepositoryInterface;

/**
 * Class DiscountServiceProvider
 * @package Mtc\MultiBuy\Providers
 */
class DiscountServiceProvider extends ServiceProvider
{
    /**
     * Bootstrap the application services.
     */
    public function boot()
    {
        // Check if plugin enabled
        if (config('discounts.enabled') !== true) {
            return;
        }

        Relation::morphMap([
            'discount' => config('discounts.discount_class')
        ]);

        // Load translations
        $this->loadTranslationsFrom(dirname(__DIR__, 2) . '/resources/lang', 'discounts');

        // Load migrations
        if ($this->app->runningInConsole()) {
            $this->loadMigrationsFrom(dirname(__DIR__, 2) . '/database/migrations');
            $this->bootResourcePublish();

            $this->commands([
                InstallDiscounts::class
            ]);
        } else {
            app('twig_loader')->addPath(dirname(__DIR__, 2) . '/resources/templates');
        }
    }

    /**
     * Register the application services.
     */
    public function register()
    {
        // Automatically apply the package configuration
        $this->mergeConfigFrom(dirname(__DIR__, 2) . '/config/discounts.php', 'discounts');

        //Check if plugin enabled
        if (config('discounts.enabled', false) !== true) {
            return;
        }

        //Register service providers
        collect(config('discounts.service_providers', []))
            ->each(function ($service_provider) {
                $this->app->register($service_provider);
            });

        $this->registerBindings();
    }

    /**
     * Register bindings used by the package
     */
    protected function registerBindings()
    {
        $this->app->singleton(DiscountRepositoryInterface::class, function ($app) {
            return $this->app->make(config('discounts.repositories.discount'));
        });

        $this->app->singleton(BasketDiscountRepositoryInterface::class, function ($app) {
            return $this->app->make(config('discounts.repositories.basket_discount'));
        });

        // Register the main class to use with the facade
        $this->app->singleton('discount', function () {
            return $this->app->make(config('discounts.discount_class'));
        });

        $this->app->singleton('discount-restriction', function () {
            return $this->app->make(config('discounts.restriction_class'));
        });
    }

    /**
     * Register resources that get published
     */
    protected function bootResourcePublish()
    {
        $this->publishes([
            dirname(__DIR__, 2) . '/config/' => config_path()
        ], 'config');

        $this->publishes([
            dirname(__DIR__, 2) . '/database/seeds/' => database_path('seeds')
        ], 'seeds');

        $this->publishes([
            dirname(__DIR__, 2) . '/resources/js/' => resource_path('js/vendor')
        ], 'assets');

        $this->publishes([
            dirname(__DIR__, 2) . '/resources/less/' => resource_path('less/vendor')
        ], 'assets');
    }
}
