<?php

namespace Mtc\MultiBuy\Repositories;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Mtc\MultiBuy\Charts\Discount as DiscountChart;
use Mtc\MultiBuy\Contracts\DiscountRepositoryInterface;

/**
 * Class DiscountRepository
 * @package Mtc\MultiBuy\Repositories
 */
class DiscountRepository implements DiscountRepositoryInterface
{
    /**
     * @var Model
     */
    protected $discount;

    /**
     * Get Discount Class
     *
     * @return Model
     */
    public function getDiscountClass()
    {
        return App::make('discount');
    }

    /**
     * Find a discount by id
     *
     * @param int $id
     */
    public function findById(int $id)
    {
        $this->discount = $this->getDiscountClass()->find($id);
    }

    /**
     * Set discount model
     *
     * @param $coupon
     */
    public function setModel($discount)
    {
        $this->discount = $discount;
    }

    /**
     * Get discount ID
     *
     * @return integer|null
     */
    public function getId()
    {
        return $this->discount ? $this->discount->id : null;
    }

    /**
     * Fetch the current discount model
     *
     * @return mixed
     */
    public function getModel()
    {
        return $this->discount ?: $this->getDiscountClass();
    }

    /**
     * Get Discount Types
     *
     * @return Collection
     */
    public function getTypes() : Collection
    {
        return collect(config('discounts.discount_types', []))
            ->map(function ($type) {
                return $type['description'];
            });
    }

    /**
     * Get all available discount restrictions
     *
     * @return Collection
     */
    public function getRestrictions() : Collection
    {
        return collect(config('discounts.restrictions'));
    }

    /**
     * Get Basic restrictions
     *
     * @return Collection
     */
    public function getBasicRestrictions() : Collection
    {
        return collect(config('discounts.basic_restrictions'))
            ->filter(function($basic_restriction) {
                return $basic_restriction['enabled'] === true;
            });
    }

    /**
     * Get Sale Restrictions
     *
     * @return Collection
     */
    public function getSaleRestrictions() : Collection
    {
        return collect(config('discounts.sale_restrictions', []));
    }

    /**
     * Get Discounts
     *
     * @param bool $paginate
     * @return mixed
     */
    public function getDiscounts(bool $paginate = true)
    {
        $query = $this->getDiscountClass()
            ->newQuery()
            ->search(request());

        return $paginate ? $query->latest()->paginate(config('discounts.admin.pagination')) : $query->get();
    }

    /**
     * Get discount usage chart
     *
     * @return DiscountChart|null
     */
    public function getChart()
    {
        return config('discounts.chart.enabled') ? new DiscountChart($this->getModel(), request()) : null;
    }

    /**
     * Update or Create Model from request
     *
     * @param $request
     * @param null $id
     */
    public function updateOrCreateModel($request, $id = null)
    {
        if ($id > 0) {
            $this->findById($id);
        }

        $validated = $request->validated();
        $this->discount = $this->getModel();
        $this->discount->fill($validated);

        $basic_restrictions = $this->getBasicRestrictions()
            ->keys()
            ->mapWithKeys(function($basic_restriction) use ($validated) {
                $array[$basic_restriction] = isset($validated[$basic_restriction]) ? $validated[$basic_restriction] : '0';
                return $array;
            });

        $this->discount->basic_restrictions = $basic_restrictions;
        $this->discount->valid_from = empty($validated['valid_from']) ? null : Carbon::createFromFormat('d/m/Y', $validated['valid_from']);
        $this->discount->valid_to = empty($validated['valid_to']) ? null : Carbon::createFromFormat('d/m/Y', $validated['valid_to']);
        $this->discount->save();
    }

    /**
     * Delete Model by Id
     *
     * @param $id
     */
    public function deleteById($id)
    {
        $this->findById($id);

        if ($this->getModel()) {
            $this->getModel()->delete();
        }
    }
}
