<?php

namespace Mtc\MultiBuy\Repositories;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\App;
use Mtc\MultiBuy\Contracts\BasketDiscountRepositoryInterface;

/**
 * Class BasketDiscountRepository
 * @package Mtc\MultiBuy\Repositories
 */
class BasketDiscountRepository implements BasketDiscountRepositoryInterface
{
    /**
     * @var Collection
     */
    protected $discounts;

    /**
     * @var Model
     */
    protected $basket;

    /**
     * @var Collection
     */
    protected $basic_restriction_validators;

    /**
     * Get Discount Class
     *
     * @return Model
     */
    public function getDiscountClass()
    {
        return App::make('discount');
    }

    /**
     * Set Basket
     *
     * @param $basket
     */
    public function setBasket($basket)
    {
        $this->basket = $basket;
    }

    /**
     * Set discounts for currect basket
     */
    public function setBasketDiscounts()
    {
        if (!$this->basket) {
            return;
        }

        $this->removeBasketDiscounts();
        $this->getBasketDiscounts();
        $this->addDiscountsToBasket();
    }

    /**
     * Add available discounts to basket
     */
    protected function addDiscountsToBasket()
    {
        $this->discounts->each(function ($discount) {
            $this->basket->discounts()
                ->create([
                    'discount_id' => $discount->id,
                    'discount_type' => 'discount',
                    'applicable_items' => $discount->items->toArray()
                ]);
        });
    }

    /**
     * Remove all basket discounts
     */
    protected function removeBasketDiscounts()
    {
        $this->basket->discounts()
            ->where('discount_type', 'discount')
            ->delete();
    }

    /**
     * Get basket available discounts
     */
    protected function getBasketDiscounts()
    {
        //Set Basket validators
        $this->setBasicRestrictionsValidators();
        $discounts = $this->getDiscountClass()
            ->active()
            ->with('restrictions')
            ->orderBy('order')
            ->get()
            ->filter(function ($discount) {
                return $this->availableForBasket($discount);
            })
            ->each(function ($discount) {
                $discount->items = $this->basket->items
                    ->filter(function($item) use ($discount) {
                        return $this->availableForItem($discount, $item);
                    })
                    ->map(function($item) {
                        return $item->purchasable->node->id;
                    });
            })
            ->filter(function ($discount) {
                return $discount->items->count();
            });

        $first_discount = $discounts->first();
        if (!$first_discount) {
            $this->discounts = collect([]);
            return;
        }
        $discount_stackables = $first_discount->stackables
            ->pluck('id')
            ->toArray();
        $this->discounts = $discounts->filter(function($discount) use ($first_discount, $discount_stackables) {
            return $discount->id === $first_discount->id || in_array($discount->id, $discount_stackables);
        });
    }

    /**
     * Get Basic restriction validators
     *
     * @return Collection
     */
    protected function setBasicRestrictionsValidators()
    {
        $this->basic_restriction_validators = collect(config('discounts.basic_restrictions'))
            ->filter(function($basic_restriction) {
                return $basic_restriction['enabled'] === true && !empty($basic_restriction['validator']);
            })->pluck('validator');
    }

    /**
     * Check if discount is available for basket
     *
     * @param $discount
     * @return bool
     */
    public function availableForBasket($discount) : bool
    {
        $failed_validators = $this->basic_restriction_validators
            ->reject(function($validator_class) use ($discount) {
                return App::make($validator_class)->validate($discount, $this->basket);
            })
            ->count();

        return !$failed_validators;
    }

    /**
     * Check if discount is available for item
     *
     * @param $discount
     * @param $item
     * @return bool
     */
    public function availableForItem($discount, $item) : bool
    {
        $failed_validators = collect(config('discounts.item_checks'))
            ->reject(function($validator_class) use ($discount, $item) {
                return App::make($validator_class)->validate($discount, $item);
            })
            ->count();

        if ($failed_validators) {
            return false;
        }

        //Check if discount have restrictions
        if ($discount->restrictions->count()) {

            $excluded_restrictions = $discount->restrictions
                ->where('exclude', true);

            $included_restrictions = $discount->restrictions
                ->where('exclude', false)
                ->groupBy('discountable_type');

            $restrictions_count = [
                'excluded' => $excluded_restrictions->count(),
                'included' => $included_restrictions->count()
            ];

            $passed_included_restrictions = $included_restrictions
                ->filter(function($restrictions, $discountable_type) use ($discount, $item, $restrictions_count) {
                    return $restrictions
                        ->filter(function($restriction) use ($discount, $item, $restrictions_count) {
                            return App::make(config('discounts.restrictions.' . $restriction->discountable_type . '.validation'))
                                ->validate($discount, $item, $restriction, $restrictions_count);
                        })
                        ->count();
                });

            $failed_included_restrictions = $included_restrictions->diffAssoc($passed_included_restrictions)
                ->map(function($restrictions, $discountable_type) {
                    return config('discounts.restrictions.' . $discountable_type . '.priority', 0);
                })
                ->sortByDesc(function($priority) {
                    return $priority;
                });

            $passed_included_restrictions = $passed_included_restrictions
                ->map(function($restrictions, $discountable_type) {
                    return config('discounts.restrictions.' . $discountable_type . '.priority', 0);
                })
                ->sortByDesc(function($priority) {
                    return $priority;
                });

            $failed_excluded_restrictions = $excluded_restrictions
                ->reject(function($restriction) use ($discount, $item, $restrictions_count) {
                    return App::make(config('discounts.restrictions.' . $restriction->discountable_type . '.validation'))
                        ->validate($discount, $item, $restriction, $restrictions_count);
                })
                ->map(function($restriction) {
                    return config('discounts.restrictions.' . $restriction->discountable_type . '.priority', 0);
                })
                ->sortByDesc(function($priority) {
                    return $priority;
                });

            $passed_excluded_restrictions = $excluded_restrictions->diffAssoc($failed_excluded_restrictions)
                ->map(function($restriction) {
                    return config('discounts.restrictions.' . $restriction->discountable_type . '.priority', 0);
                })
                ->sortByDesc(function($priority) {
                    return $priority;
                });

            if ($restrictions_count['excluded'] === 0) {
                if ($failed_included_restrictions->count() && $passed_included_restrictions->count()) {
                    if ($failed_included_restrictions->first() > $passed_included_restrictions->first()) {
                        return true;
                    }
                }
            }

            if ($failed_excluded_restrictions->count() === 0) {
                if ($passed_included_restrictions->count() && $passed_excluded_restrictions->count()) {
                    if ($passed_included_restrictions->first() < $passed_excluded_restrictions->first()) {
                        return true;
                    }
                }

                if ($failed_included_restrictions->count() && $passed_excluded_restrictions->count()) {
                    if ($failed_included_restrictions->first() > $passed_excluded_restrictions->first()) {
                        return true;
                    }
                }
            }

            if ($passed_included_restrictions->count() !== $restrictions_count['included']) {
                return false;
            }

            if ($failed_excluded_restrictions->count()) {
                if ($passed_included_restrictions->count()) {
                    if ($failed_excluded_restrictions->first() > $passed_included_restrictions->first()) {
                        return false;
                    }
                } else {
                    return false;
                }
            }
        }

        return true;
    }

    /**
     * Get discount applicable items for current basket
     *
     * @param $discount
     * @return Collection|mixed
     */
    public function getDiscountApplicableItems($discount)
    {
        if (!$this->basket) {
            return new Collection();
        }

        $basket_discount = $this->basket
            ->discounts
            ->where('discount_id', $discount->id)
            ->where('discount_type', 'discount')
            ->first();

        if (!$basket_discount) {
            return new Collection();
        }

        //Get basket discount applicable items
        $applicable_items = $basket_discount->applicable_items;

        return $this->basket->items
            ->filter(function ($item) use ($applicable_items) {
                //Check if item is in discountables list
                return in_array($item->purchasable->node->id, $applicable_items);
            });

    }
}
