<?php

namespace Mtc\MtcPay;

use Illuminate\Http\Request;
use Mtc\Checkout\Contracts\PaymentGateway;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Mtc\Checkout\Contracts\PayableContract;
use Mtc\Checkout\Contracts\HandlesRefunds;
use Mtc\Checkout\Invoice\Payment;
use Mtc\Checkout\PaymentForm;
use Illuminate\Support\Facades\URL;
use Mtc\MtcPay\Exceptions\ChargeException;

/**
 * MtcPay Payment Gateway
 *
 * @package  Mtc\MtcPay
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class MtcPay implements PaymentGateway, HandlesRefunds
{
    /**
     * Version for debug/telemetry on stripe
     */
    const VERSION = 'plat_1.0';

    /**
     * Check if the gateway is available for use on this payment.
     *
     * @param InvoiceRepositoryContract $invoice
     * @param PayableContract $payable
     * @return bool
     */
    public function isApplicable(InvoiceRepositoryContract $invoice, $payable): bool
    {
        return $invoice->getOutstandingAmount() > 0.3;
    }

    /**
     * Render the payment template.
     *
     * @param InvoiceRepositoryContract $invoice
     * @return string
     */
    public function getPaymentForm(InvoiceRepositoryContract $invoice): PaymentForm
    {
        return new PaymentForm('mtc-pay-form', 'vue-component', [
            'name' => __('mtc_pay::checkout.name'),
            'button_text' => __('mtc_pay::checkout.button'),
            'button_text_processing' => __('mtc_pay::checkout.button_processing'),
            'checkout_url' => URL::signedRoute('checkout.mtc-pay.request', ['invoice_id' => $invoice->getId()]),
            'invoice_id' => $invoice->getId(),
        ]);
    }

    /**
     * Charge payment on invoice
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @throws ChargeException
     * @return array
     */
    public function charge(Request $request, InvoiceRepositoryContract $invoice): array
    {
        $response = (new ApiClient)->call('payment_intent/' . $request->query('payment_intent'));
        if (empty($response['success'])) {
            throw new ChargeException('Error retrieving intent');
        }

        if ($response['payment_intent']['client_secret'] !== $request->query('payment_intent_client_secret')) {
            throw new ChargeException('Client secret mismatch');
        }

        if ($response['payment_intent']['status'] !== 'succeeded') {
            throw new ChargeException('Transaction failed. Status is ' . $response['payment_intent']['status']);
        }

        return [
            'provider' => 'mtc_pay',
            'amount' => $response['payment_intent']['amount'] / 100,
            'currency_code' => $invoice->getCurrency(),
            'amount_in_currency' => $invoice->getOutstandingAmountInCurrency(),
            'reference' => $response['payment_intent']['id'],
            'details' => $response,
            'confirmed_at' => now(),
        ];
    }

    /**
     * Check if a payment is refundable by the payment gateway
     *
     * @param Payment $payment
     * @return bool|array
     */
    public function isRefundable(Payment $payment)
    {
        try {
            $response = (new ApiClient)->call('payment_intent/' . $payment->reference);
            $charge = $response['payment_intent']['charges']['data'][0];

            if ($charge['refunds']['total_count'] > 0) {
                $payment->refundable_amount = ($charge['amount'] - $charge['amount_refunded']) / 100;
            }

            return $charge['refunded'] == false;
        } catch (\Exception $e) {
            return false;
        }


    }
    
    /**
     * Process a refund on a payment
     *
     * @param Payment $payment
     * @param null|double $amount
     * @return bool|array
     */
    public function refund(Payment $payment, $amount = null)
    {
        $payment_intent = $payment->reference;
        $amount = ($amount ?? $payment->refundable_amount) * 100;
        try {
            $response = (new ApiClient)->call('refund', 'POST', compact('payment_intent', 'amount'));
            if (empty($response['success'])) {
                throw new RefundException("Issue with refund: " . $response['message']);
            }
        } catch (Exception $e) {
            return false;
        }

        return [
            'reference' => $response['refund']['id'],
            'amount' => $amount / 100,
        ];
    }
}