<?php

namespace Mtc\MtcPay\Http\Controllers;

use Illuminate\Http\Request;
use Mtc\Checkout\Contracts\InvoiceRepositoryContract;
use Illuminate\Support\Facades\URL;
use Mtc\MtcPay\ApiClient;
use Mtc\MtcPay\CheckoutFactory;

/**
 * Controller for comms between Vue Component & website.
 *
 * @package  Mtc\MtcPay
 * @author   Craig McCreath <craig.mccreath@mtcmedia.co.uk>
 */
class MtcPayController
{
    /**
     * From the Vue Component, request a new MTC Pay checkout and provide information needed to set up Stripe Elements.
     *
     * @param Request $request
     * @param InvoiceRepositoryContract $invoice
     * @return array|\Illuminate\Contracts\Routing\ResponseFactory
     * @throws \Illuminate\Contracts\Container\BindingResolutionException
     */
    public function index(Request $request, InvoiceRepositoryContract $invoice) {

        // We are using a signed route to ensure folks don't try and access other invoices
        if (false === $request->hasValidSignature()) {
            return response([
                'success' => 'false',
                'error' => 'Invalid signature',
            ], 401);
        }

        $invoice->load($request->query('invoice_id'));

        if ($issues = $this->invoiceHasIssues($invoice)) {
            return $issues;
        }

        try {
            $request_data = app()->make(config('mtc_pay.checkout_factory_class'))($invoice);
            $checkout = (new ApiClient)->call('checkout', 'POST', $request_data);

            return [
                'success' => true,
                'data' => [
                    'publishable_key' => $checkout['publishable_key'],
                    'payment_intent_id' => $checkout['payment_intent']['id'],
                    'client_secret' => $checkout['payment_intent']['client_secret'],
                    'test_mode' => str_contains($checkout['publishable_key'], '_test_'),
                    'return_url' => $request_data['return_url'],
                    'default_values' => [
                        'billing_details' => $request_data['billing_details'],
                    ],
                ],
            ];

        } catch (\Exception $e) {
            if (config('app.debug')) throw $e;

            return response([
                'success' => false,
                'error' => [$e->getMessage()]
            ], 422);
        }
    }

    /**
     * Verify the current invoice doesn't have any issues before proceeding with a new checkout.
     *
     * @param InvoiceRepositoryContract $invoice
     * @return array
     */
    protected function invoiceHasIssues(InvoiceRepositoryContract $invoice): array
    {
        if (is_null($invoice->getModel())) {
            return [
                'success' => false,
                'issues' => ['Unable to load invoice'],
                'redirect' => route('checkout.index'),
            ];
        }

        if ($invoice->hasConflicts()) {
            return [
                'success' => false,
                'issues' => $invoice->getConflicts(),
                'redirect' => route('checkout.index'),
            ];
        }

        if ($invoice->isPaid()) {
            return [
                'success' => true,
                'redirect' => URL::signedRoute('successful_payment', ['id' => $invoice->getId() ], null, config('app.env') === 'production'),
            ];
        }

        return [];
    }
}