<?php

namespace Mtc\Login;

use Illuminate\Contracts\Auth\StatefulGuard;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\App;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Str;
use Illuminate\Validation\ValidationException;
use Laravel\Fortify\Fortify;
use Laravel\Fortify\LoginRateLimiter;
use Mtc\Login\Jobs\CheckOldStaff;
use Mtc\Login\Listeners\ValidateMtcLogin;

class MtcUserVerifyInFortify
{
    /**
     * The credentials provided by the Attempting event.
     *
     * @var array
     */
    protected $credentials = [];

    /**
     * The response from the API.
     *
     * @var null|array
     */
    protected $response;

    /**
     * The user model.
     *
     * @var Model Filled during findLocalUser()
     */
    protected $user;

    /**
     * ValidateMtcLogin constructor.
     *
     * @param LoginService $service
     */
    public function __construct(LoginService $service)
    {
        $this->service = $service;
    }

    /**
     * Handle the incoming request.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  callable  $next
     * @return mixed
     */
    public function handle($request, $next)
    {
        $this->credentials = $request->only(Fortify::username(), 'password');
        if (! $this->isMtc()) {
            return $next($request);
        }

        $this->findLocalUser()
            ->verifyUser()
            ->updateUser();

        return $next($request);
    }

    /**
     * Determine if the email provided is our own.
     *
     * @return bool
     */
    protected function isMtc()
    {
        return Str::contains($this->credentials[Fortify::username()], '@mtcmedia.co.uk')
            && !Str::contains($this->credentials[Fortify::username()], 'dev.');
    }

    /**
     * Find if the user exists and if not, return a new model.
     *
     * @return self
     */
    protected function findLocalUser()
    {
        $this->user = App::make(config('auth.providers.users.model'))->newQuery()
            ->where(Fortify::username(), $this->credentials[Fortify::username()])
            ->firstOrNew([]);

        return $this;
    }

    /**
     * Run a call against the API and store the decoded JSON response.
     *
     * @return self
     */
    protected function verifyUser()
    {
        $this->response = $this->service->isValidUser($this->credentials[Fortify::username()], $this->credentials['password']);

        if (!empty($this->response) && $this->wonStaffCheckLottery()) {
            dispatch(new CheckOldStaff);
        }
        return $this;
    }

    /**
     * Update the user with the results of the response.
     *
     * @return void
     */
    protected function updateUser()
    {
        // No need to continue if user doesn't exist and bad response.
        if ($this->userWasNotVerified()) {
            return;
        }

        // Update pass to a random string if we have a bad response.
        $this->user->password = $this->response === null
            ? bcrypt(Str::random(40))
            : bcrypt($this->credentials['password']);

        // Add the rest of the params if the user doesn't exist already.
        if (!$this->user->exists) {
            $this->user->fill($this->response);
            $this->user->name = $this->response['name'];
        }

        $this->user->save();

        // Ensure mtc. users are assigned to the master role .
        if ($this->shouldAssignMasterRole()) {
            $this->user->assignRole(config('mtc_login.mtc_user_role'));
        }
    }

    /**
     * Check if user was verified
     *
     * @return bool
     */
    protected function userWasNotVerified()
    {
        return $this->response === false
            || ($this->user->exists == false && $this->response === null);
    }

    /**
     * Check if role should be assigned to user
     * This ensures role is set, users have roles and user doesn't have it already
     *
     * @return bool
     */
    protected function shouldAssignMasterRole()
    {
        return method_exists($this->user, 'hasRole')
            && !empty(config('mtc_login.mtc_user_role'))
            && $this->user->hasRole(config('mtc_login.mtc_user_role')) === false;
    }

    /**
     * Check if should dispatch the staff check.
     * A lottery to trigger this to avoid calling this too often
     *
     * @return bool
     */
    protected function wonStaffCheckLottery()
    {
        try {
            return random_int(0, config('mtc_login.old_check_lottery_range')) == 1;
        } catch (\Exception $e) {
            return false;
        }
    }
}
