<?php

namespace Mtc\MotorCheck\Maps;

use Carbon\Carbon;
use Illuminate\Support\Str;
use Mtc\AutomotiveSpecs\VehicleSpecification;
use Mtc\MotorCheck\Contracts\MotorSpecsToVehicleSpecificationContract;

class MotorSpecsToVehicleSpecification implements MotorSpecsToVehicleSpecificationContract
{
    protected array $data;

    public function __construct(protected VehicleSpecification $spec)
    {
    }

    public function map(array $spec_data): VehicleSpecification
    {
        $this->data = $spec_data;
        $this->setBasicValues();
        $this->setFeatureList();
        return $this->spec;
    }

    protected function setBasicValues(): void
    {
        $this->spec->setValuesFromArray([
            'UUID' => $this->data['identity']['vehicleId'],
            'registration_number' => $this->data['identity']['vehicle']['combined']['registration'] ??
                    $this->data['identity']['vehicle']['standard']['registration'] ??
                    $this->data['identity']['registration'],
            'make' => $this->make(),
            'model' => $this->model(),
            'derivative' => $this->derivative(),
            'model_year' => $this->modelYear(),
            'registration_date' => $this->registrationDate(),
            'engine_size' => $this->engineSize(),
            'mpg' => $this->mpg(),
            'co2' => $this->co2(),
            'body_style' => $this->bodyStyle(),
            'colour' => $this->colour(),
            'transmission' => $this->transmission(),
            'fuel_type' => $this->fuelType(),
            'drivetrain' => $this->drivetrain(),
            'odometer_value' => $this->odometerValue(),
            'odometer_metric' => $this->odometerMetric(),
            'vin' => $this->vin(),
            'doors' => $this->doors(),
            'seats' => $this->seats(),
            'previous_owner_count' => $this->previousOwners(),
        ]);
    }

    protected function setFeatureList(): void
    {
        $feature_list = collect($this->data['standard']['standardSpecification'] ?? [])
            ->map(function ($category_group) {
                return collect($category_group)
                    ->map(function ($category_items, $category) {
                        $category_items = collect($category_items)
                            ->map(function ($category_item) use ($category) {
                                return [
                                    'name' => $category_item['name'],
                                    'value' => $category_item['description'],
                                    'type' => $category
                                ];
                            });
                        return $category_items;
                    });
            })
            ->flatten(2);
        $this->spec->setFeatureList($feature_list->toArray());
    }

    protected function make(): string
    {
        $make = $this->data['identity']['vehicle']['nvdf']['make'] ??
            $this->data['identity']['specsVehicle']['make'] ?? '';
        return Str::title($make);
    }

    protected function model(): string
    {
        $value = $this->data['identity']['vehicle']['nvdf']['model'] ??
            $this->data['identity']['specsVehicle']['model'] ?? '';
        return Str::title($value);
    }

    protected function modelYear(): int
    {
        $year = $this->data['identity']['vehicle']['nvdf']['year'] ?? null;
        if ($year) {
            return $year;
        }

        try {
            $reg_date = $this->registrationDate();
            if (!empty($reg_date)) {
                return Carbon::parse($reg_date)->year;
            }
        } catch (\Exception $exception) {
        }
        return $this->data['identity']['specsVehicle']['modelYear'] ?? 0;
    }

    protected function derivative(): string
    {
        if (!empty($this->data['identity']['vehicle']['nvdf']['description'])) {
            return $this->data['identity']['vehicle']['nvdf']['description'];
        }

        if (!empty($this->data['identity']['vehicle']['mvris']['vehicleDesc'])) {
            return $this->data['identity']['vehicle']['mvris']['vehicleDesc'];
        }

        if (!empty($this->data['identity']['vehicle']['combined']['version'])) {
            return $this->data['identity']['vehicle']['combined']['version'];
        }

        if (!empty($this->data['identity']['specsVehicle']['version'])) {
            return $this->data['identity']['specsVehicle']['version'];
        }

        return '';
    }

    protected function registrationDate(): string
    {
        return $this->data['identity']['vehicle']['combined']['regDate'] ??
            $this->data['identity']['vehicle']['mvris']['regDate'] ??
            $this->data['identity']['vehicle']['nvdf']['regDate'] ??
            $this->data['identity']['vehicle']['dvla']['regDate'] ?? '';
    }

    protected function engineSize(): int
    {
        return (int)($this->data['identity']['vehicle']['nvdf']['cc'] ??
            $this->data['identity']['vehicle']['combined']['cc'] ??
            $this->data['identity']['vehicle']['mvris']['cc'] ??
            $this->data['identity']['vehicle']['dvla']['cc'] ??
            $this->data['identity']['specsVehicle']['engineCC'] ?? 0);
    }

    protected function mpg(): int
    {
        return (int)($this->data['identity']['vehicle']['mvris']['combinedMpg'] ?? 0);
    }

    protected function co2(): int
    {
        return (int)($this->data['identity']['vehicle']['dvla']['co2'] ??
            $this->data['identity']['vehicle']['nvdf']['co2'] ?? 0);
    }

    protected function bodyStyle(): string
    {
        return Str::title($this->data['identity']['vehicle']['combined']['body'] ??
            $this->data['identity']['vehicle']['mvris']['body'] ??
            $this->data['identity']['vehicle']['nvdf']['body'] ??
            $this->data['identity']['vehicle']['dvla']['body'] ??
            $this->data['identity']['specsVehicle']['body'] ?? '');
    }

    protected function vin(): string
    {
        return $this->data['identity']['vehicle']['combined']['vin'] ??
            $this->data['identity']['vehicle']['nvdf']['vin'] ??
            $this->data['identity']['vehicle']['dvla']['vin'] ?? '';
    }

    protected function doors(): int
    {
        return (int)($this->data['identity']['vehicle']['combined']['doors'] ??
            $this->data['identity']['vehicle']['mvris']['doors'] ??
            $this->data['identity']['vehicle']['nvdf']['doors'] ??
            $this->data['identity']['vehicle']['dvla']['doors'] ??
            $this->data['identity']['specsVehicle']['doors'] ?? 0);
    }

    protected function seats(): int
    {
        return (int)($this->data['identity']['vehicle']['mvris']['seatCount'] ??
            $this->data['identity']['vehicle']['nvdf']['seatingCapacity'] ?? 0);
    }

    protected function previousOwners(): int
    {
        return $this->data['identity']['vehicle']['keepers']['numberOfPrevious'] ?? 0;
    }

    protected function colour(): string
    {
        return Str::title($this->data['identity']['vehicle']['combined']['colour'] ??
            $this->data['identity']['vehicle']['mvris']['colour'] ??
            $this->data['identity']['vehicle']['nvdf']['colour'] ??
            $this->data['identity']['vehicle']['dvla']['colour'] ??
            $this->data['identity']['specsVehicle']['colour'] ?? '');
    }

    protected function transmission(): string
    {
        return Str::title($this->data['identity']['specsVehicle']['transmission'] ??
            $this->data['identity']['vehicle']['combined']['transmission'] ??
            $this->data['identity']['vehicle']['mvris']['transmission'] ??
            $this->data['identity']['vehicle']['nvdf']['transmission'] ??
            $this->data['identity']['vehicle']['dvla']['transmission'] ?? '');
    }

    protected function fuelType(): string
    {
        return Str::title($this->data['identity']['vehicle']['combined']['fuel'] ??
            $this->data['identity']['vehicle']['mvris']['fuel'] ??
            $this->data['identity']['vehicle']['nvdf']['fuel'] ??
            $this->data['identity']['vehicle']['dvla']['fuel'] ??
            $this->data['identity']['specsVehicle']['fuel'] ?? '');
    }

    protected function drivetrain(): string
    {
        return $this->data['identity']['vehicle']['mvris']['driveType'] ?? '';
    }

    protected function odometerValue(): int
    {
        return (int)($this->data['identity']['vehicle']['combined']['currentMiles'] ?? 0);
    }

    protected function odometerMetric(): string
    {
        return 'mi';
    }
}
