<?php

namespace Tests\Feature;

use Illuminate\Foundation\Testing\DatabaseMigrations;
use Illuminate\Support\Facades\Http;
use Mtc\AutomotiveSpecs\Facades\AutoSpecs;
use Mtc\AutomotiveSpecs\VehicleRisks;
use Mtc\AutomotiveSpecs\VehicleSpecification;
use Mtc\AutomotiveSpecs\VehicleValuation;
use Mtc\MotorCheck\Exceptions\NotFoundException;
use Mtc\MotorCheck\Models\MotorCheckRisk;
use Mtc\MotorCheck\Models\MotorCheckSpec;
use Mtc\MotorCheck\Models\MotorCheckValuation;
use Mtc\Tests\TestCase;

class MotorCheckServiceTest extends TestCase
{
    use DatabaseMigrations;

    protected function getEnvironmentSetUp($app)
    {
        $app['config']->set('automotive_specs.driver', 'motor-check');
        Http::fake([
            "https://staging.motorspecs.com/oauth" =>  Http::response(['access_token' => 'test-token']),
            "https://trade-api.motorcheck.co.uk/api/v1/oauth/token" =>  Http::response(['access_token' => 'mc-token']),
            "https://staging.motorspecs.com/identity-specs/lookup" =>  Http::response($this->specData()),
            "https://staging.motorspecs.com/specs/standard" =>  Http::response($this->standardData()),
            "https://staging.motorspecs.com/valuation-autotrader/value" =>  Http::response($this->valuationDataAT()),
            "https://staging.motorspecs.com/valuation-vip/value" =>  Http::response($this->valuationDataVIP()),
            "https://trade-api.motorcheck.co.uk/api/v1/report/search" =>  Http::response($this->reportSearchData()),
            "https://trade-api.motorcheck.co.uk/api/v1/report/show" =>  Http::response($this->reportData()),
            '*' => Http::response(['result' => 'wildcard'])
        ]);
    }

    public function testMissingVrmOnSpecs()
    {
        $this->expectException(NotFoundException::class);
        AutoSpecs::getCarSpecs('');
    }

    public function testMissingVrmOnValuation()
    {
        $this->expectException(NotFoundException::class);
        AutoSpecs::getValuation('');
    }

    public function testMissingVrmOnRisks()
    {
        $this->expectException(NotFoundException::class);
        AutoSpecs::getCarRisks('');
    }

    public function testGetRawSpecs()
    {
        $this->app['config']->set('motor_check.allow_local_cache', false);
        $spec = AutoSpecs::getCarSpecs('HY63CBO');

        self::assertInstanceOf(VehicleSpecification::class, $spec);
        self::assertEquals('Volkswagen', $spec->make);
        self::assertEquals('Golf', $spec->model);
        self::assertEquals(2014, $spec->model_year);
        self::assertEquals('SE TDI 1.6 105 PS', $spec->derivative);
        self::assertEquals(1598, $spec->engine_size);
        self::assertEquals(99, $spec->co2);
        self::assertEquals('Hatchback', $spec->body_style);
        self::assertEquals('Manual', $spec->transmission);
        self::assertEquals('Diesel', $spec->fuel_type);
        self::assertEquals('White', $spec->colour);
        self::assertEquals('i9dp3EGuLNhRL9jxQ+0GkyrD3XANI46BehA7ROIY8uA=', $spec->UUID);
        self::assertEquals('2013-09-14', $spec->registration_date);
        self::assertEquals('HY63CBO', $spec->registration_number);
        self::assertEquals('mi', $spec->odometer_metric);
        self::assertEquals('80000', $spec->odometer_value);
        self::assertEquals(72, $spec->mpg);
        self::assertEquals('4X2', $spec->drivetrain);
        self::assertEquals(118, count($spec->feature_list));
    }

    public function testGetRawValuationAutoTrader()
    {
        $this->app['config']->set('motor_check.allow_local_cache', false);
        $this->app['config']->set('motor_check.valuation_driver', 'auto-trader');

        $valuation = AutoSpecs::getValuation('HY63CBO');

        self::assertInstanceOf(VehicleValuation::class, $valuation);
        self::assertEquals(9975, $valuation->retail);
        self::assertEquals(7050, $valuation->part_exchange);
    }

    public function testGetRawValuationVip()
    {
        $this->app['config']->set('motor_check.allow_local_cache', false);
        $this->app['config']->set('motor_check.valuation_driver', 'vip');
        $valuation = AutoSpecs::getValuation('HY63CBO');

        self::assertInstanceOf(VehicleValuation::class, $valuation);
        self::assertEquals(6370, $valuation->retail);
        self::assertEquals(5460, $valuation->part_exchange);
    }

    public function testGetRawRisks()
    {
        $this->app['config']->set('motor_check.allow_local_cache', false);
        $risks = AutoSpecs::getCarRisks('HY63CBO');

        self::assertInstanceOf(VehicleRisks::class, $risks);
        self::assertEquals('Finance agreement in place', $risks->finance);
        self::assertEquals(false, $risks->mileage);
        self::assertEquals(false, $risks->at_risk);
        self::assertEquals(false, $risks->plate_change);
        self::assertEquals(false, $risks->colour_changes);
        self::assertEquals(false, $risks->tax_and_sorn);
        self::assertEquals(false, $risks->origin_and_use);
        self::assertEquals(false, $risks->keepers);
        self::assertEquals(false, $risks->stolen);
        self::assertEquals(false, $risks->write_off);
        self::assertEquals(false, $risks->condition);
        self::assertEquals(false, $risks->scrapped);
        self::assertEquals(false, $risks->mot_history);

    }
    public function testCacheSpecs()
    {
        $this->app['config']->set('motor_check.allow_local_cache', true);
        AutoSpecs::getCarSpecs('HY63CBO');

        self::assertTrue(MotorCheckSpec::query()->where('vrm', 'HY63CBO')->exists());
    }

    public function testCacheValuation()
    {
        $this->app['config']->set('motor_check.allow_local_cache', true);
        AutoSpecs::getValuation('HY63CBO');

        self::assertTrue(MotorCheckValuation::query()->where('vrm', 'HY63CBO')->exists());
    }

    public function testCacheRisks()
    {
        $this->app['config']->set('motor_check.allow_local_cache', true);
        AutoSpecs::getCarRisks('HY63CBO');

        self::assertTrue(MotorCheckRisk::query()->where('vrm', 'HY63CBO')->exists());
    }

    public function testGetCachedSpecs()
    {
        MotorCheckSpec::query()
            ->create([
                'vrm' => 'HY63CBO',
                'mileage' => 0,
                'identity' => $this->specData(),
                'standard' => $this->standardData()
            ]);

        $spec = AutoSpecs::getCarSpecs('HY63CBO');

        self::assertInstanceOf(VehicleSpecification::class, $spec);
        self::assertEquals('Volkswagen', $spec->make);
        self::assertEquals('Golf', $spec->model);
        self::assertEquals(2014, $spec->model_year);
        self::assertEquals(118, count($spec->feature_list));
    }

    public function testGetCachedValuation()
    {
        $this->app['config']->set('motor_check.valuation_driver', 'auto-trader');
        MotorCheckValuation::query()
            ->create([
                'vrm' => 'HY63CBO',
                'autotraderValuation' => $this->valuationDataAT()['autotraderValuation']
            ]);

        $valuation = AutoSpecs::getValuation('HY63CBO');

        self::assertInstanceOf(VehicleValuation::class, $valuation);
        self::assertEquals(9975, $valuation->retail);
        self::assertEquals(7050, $valuation->part_exchange);
    }

    public function testGetCachedRisks()
    {
        MotorCheckRisk::query()
            ->create([
                'vrm' => 'HY63CBO',
                'data' => $this->reportData()['data']

            ]);

        $risks = AutoSpecs::getCarRisks('HY63CBO');

        self::assertInstanceOf(VehicleRisks::class, $risks);
        self::assertEquals('Finance agreement in place', $risks->finance);
        self::assertEquals(false, $risks->mileage);
        self::assertEquals(false, $risks->at_risk);
    }

    /**
     * https://documenter.getpostman.com/view/313931/SzKWuxGg?version=latest#b35b75a2-0528-42c9-a2b7-9cae13afb51c
     *
     * @return array
     */
    protected function specData()
    {
        return [
            'id' => 1,
            'registration' => 'HY63CBO',
            'vehicleId' => 'i9dp3EGuLNhRL9jxQ+0GkyrD3XANI46BehA7ROIY8uA=',
            'tracksTourURL' => NULL,
            'vehicle' => [
                'dvla' => [
                    'make' => 'VOLKSWAGEN',
                    'model' => 'GOLF SE BLUEMOTION TECH TDI',
                    'regDate' => '2013-09-14',
                    'v5cDate' => '2018-08-16',
                    'vin' => 'WVWZZZAUZEW080398',
                    'colour' => 'WHITE',
                    'source' => 'UK',
                    'exported' => 0,
                    'exportedDate' => NULL,
                    'imported' => 0,
                    'importedNi' => 0,
                    'importedDate' => NULL,
                    'body' => 'HATCHBACK',
                    'bodyClass' => 'CAR',
                    'seatingCapacity' => 5,
                    'wheelplan' => '2 AXLE RIGID BODY',
                    'co2' => 99,
                    'cc' => 1598,
                    'engineNumber' => 'CLHA346534',
                    'engineCode' => '1.6 TDI 105PS BLUEMOTION',
                    'fuel' => 'DIESEL',
                    'maxPower' => 77,
                    'maxTechMass' => 1820,
                    'powerToWeight' => NULL,
                    'trailerBraked' => 1500,
                    'trailerUnbraked' => 660,
                    'grossWeight' => 1820,
                ],
                'keepers' => [
                    'numberOfPrevious' => 2,
                    'startDate' => '2018-07-28',
                    'previousAcquire' => '2015-12-15',
                    'previousDispose' => '2018-07-02',
                ],
                'mvris' => [
                    'make' => 'VOLKSWAGEN',
                    'model' => 'GOLF',
                    'vehicleDesc' => 'GOLF SE BLUEMOTION TECH TDI',
                    'fuel' => 'DIESEL',
                    'modelVariantName' => 'SE TDI BLUEMOTION TECHNOLOGY',
                    'engineSize' => 1.6,
                    'cc' => 1598,
                    'bhpCount' => 103,
                    'doorCount' => 5,
                    'bodyDesc' => 'HATCHBACK',
                    'cabType' => 'NA',
                    'gearboxType' => 'MANUAL',
                    'numberOfAxles' => 2,
                    'mvrisCode' => NULL,
                    'smmtMarketSectorLine' => NULL,
                    'vehicleSeries' => 'MK7',
                    'vehicleOrigin' => 'GERMANY',
                    'gearsCount' => 6,
                    'driveType' => '4X2',
                    'visibilityDate' => '2012-11-01',
                    'setupDate' => '2012-11-01',
                    'terminationDate' => NULL,
                    'grossCombinedWeight' => NULL,
                    'vehicleGrossWeight' => NULL,
                    'unladenWeight' => NULL,
                    'bodyShapeRoofHeight' => 'NA',
                    'driveAxle' => 'FRONT',
                    'vehicleHeight' => 1476,
                    'kerbWeight' => 1354,
                    'vehicleLength' => 4349,
                    'seatCount' => 5,
                    'powerDelivery' => 'NORMAL',
                    'rigidArtic' => 'RIGID',
                    'typeApprovalCategory' => 'M1',
                    'wheelbaseLength' => 'SHORT WHEELBASE',
                    'vehicleWidth' => 2027,
                    'primaryFuel' => 'Y',
                    'powerKw' => 77,
                    'rpmPower' => 3000,
                    'torqueLb' => 185,
                    'torqueNm' => 250,
                    'rpmTorque' => 1500,
                    'vehicleCo2' => 102,
                    'urbanColdMpg' => 61.4,
                    'extraUrbanMpg' => 80.7,
                    'combinedMpg' => 72.4,
                    'urbanColdLkm' => 4.6,
                    'extraUrbanLkm' => 3.5,
                    'combinedLkm' => 3.9,
                    'maxSpeedMph' => 119,
                    'maxSpeedKph' => 192,
                    'accelerationMph' => 10.7,
                    'accelerationKph' => 10.7,
                    'engineDescription' => '1.6 TDI 105PS BLUEMOTION',
                    'engineLocation' => 'FRONT',
                    'engineMake' => 'VOLKSWAGEN CARS',
                    'euroStatus' => 5,
                    'bore' => 76,
                    'stroke' => 87,
                    'fuelDelivery' => 'TURBO',
                    'cylinderArrangement' => 'INLINE',
                    'cylinderCount' => 4,
                    'valveCount' => 4,
                    'valveGear' => 'SOHC',
                    'regDate' => '2013-09-14',
                ],
                'combined' => [
                    'id' => 'HY63CBO',
                    'registration' => 'HY63CBO',
                    'vin' => 'WVWZZZAUZEW080398',
                    'currentMiles' => 80000,
                    'annualMiles' => 12712,
                    'make' => 'VOLKSWAGEN',
                    'model' => 'GOLF',
                    'transmission' => 'MANUAL',
                    'version' => 'GOLF SE BLUEMOTION TECH TDI',
                    'fuel' => 'DIESEL',
                    'body' => 'HATCHBACK',
                    'colour' => 'WHITE',
                    'doors' => 5,
                    'cc' => 1598,
                    'origin' => 'UK',
                    'powerBHP' => 103,
                    'powerKW' => 77,
                    'regDate' => '2013-09-14',
                    'manufDate' => '2013-09-14',
                    'imported' => false,
                    'updatedAt' => '2020-01-01 12:06:06',
                ],
            ],
            'specsVehicle' => [
                'id' => 'i9dp3EGuLNhRL9jxQ+0GkyrD3XANI46BehA7ROIY8uA=',
                'jId' => 77708620130601,
                'groupId' => NULL,
                'make' => 'VOLKSWAGEN',
                'model' => 'Golf',
                'modelYear' => 2014,
                'modelGeneration' => 7,
                'version' => 'SE TDI 1.6 105 PS',
                'engineCC' => 1598,
                'body' => 'HATCHBACK',
                'doors' => 5,
                'capacityDescriptor' => 1.6,
                'fuel' => 'DIESEL',
                'fuelPrimary' => 'DIESEL',
                'fuelSecondary' => NULL,
                'transmission' => 'MANUAL',
                'transmissionDescription' => 'manual',
                'transmissionMode' => 'MANUAL',
                'transmissionModeDesc' => NULL,
                'powertrain' => 'combustion',
                'trim' => 'SE',
                'score' => 197,
                'powerKW' => 77,
                'powerBHP' => 105,
                'introDate' => '2013-06-01',
                'concludeDate' => '2013-11-03',
                'globalSegment' => 'EU C1 - lower medium -',
                'localSegment' => 'JG Lower Medium',
            ],
            'similarVehicles' => [
                0 => [
                    'id' => 'igdi+eKHVLMB8pxWGPzVQ/wBFQscrifGqJO7apgghkk=',
                    'jId' => 777419920130601,
                    'groupId' => NULL,
                    'make' => 'VOLKSWAGEN',
                    'model' => 'Golf',
                    'modelYear' => 2014,
                    'modelGeneration' => 7,
                    'version' => 'SE TDI 1.6 105 PS DSG',
                    'engineCC' => 1598,
                    'body' => 'HATCHBACK',
                    'doors' => 5,
                    'capacityDescriptor' => 1.6,
                    'fuel' => 'DIESEL',
                    'fuelPrimary' => 'DIESEL',
                    'fuelSecondary' => NULL,
                    'transmission' => 'MANUAL',
                    'transmissionDescription' => 'dual clutch man sequential auto mode',
                    'transmissionMode' => 'AUTOMATIC',
                    'transmissionModeDesc' => 'automatic mode - manual',
                    'powertrain' => 'combustion',
                    'trim' => 'SE',
                    'score' => 197,
                    'powerKW' => 77,
                    'powerBHP' => 105,
                    'introDate' => '2013-06-01',
                    'concludeDate' => '2013-11-03',
                    'globalSegment' => 'EU C1 - lower medium -',
                    'localSegment' => 'JG Lower Medium',
                ],
            ],
            'serviceVersion' => 'V2',
            '_links' => [
                'self' => [
                    'href' => 'https://staging.motorspecs.com/identity-specs/lookup/1',
                ],
            ],
        ];
    }

    /**
     * https://documenter.getpostman.com/view/313931/SzKWuxGg?version=latest#d8e63f9d-05c6-4f80-a73c-dc2d73094e68
     * @return array
     */
    protected function standardData()
    {
        return [
            'id' => 1,
            'registration' => 'HY63CBO',
            'vehicleId' => 'i9dp3EGuLNhRL9jxQ+0GkyrD3XANI46BehA7ROIY8uA=',
            'priceData' => [
                'priceDate' => '2013-06-01',
                'priceDateFormated' => '06/2013',
                'priceDateMonth' => '06',
                'priceDateYear' => 2013,
                'price' => 20105,
                'msrp' => 20655,
                'priceOtr' => 20710,
                'deliveryCharge' => 605,
                'currency' => 'GBP',
                'currencySymbol' => '&pound;',
            ],
            'topFeatures' => [
                0 => [
                    'id' => 1401,
                    'name' => 'Remote audio controls',
                    'description' => 'Steering wheel mounted remote audio controls',
                    'value' => 'standard',
                ],
                1 => [
                    'id' => 3201,
                    'name' => 'ABS',
                    'description' => 'ABS',
                    'value' => 'standard',
                ],
                2 => [
                    'id' => 4001,
                    'name' => 'Remote boot/hatch/rear door release',
                    'description' => 'Remote controlled remote boot/hatch/rear door release',
                    'value' => 'standard',
                ],
                3 => [
                    'id' => 4501,
                    'name' => 'Cruise control',
                    'description' => 'Adaptive cruise control',
                    'value' => 'standard',
                ],
                4 => [
                    'id' => 6901,
                    'name' => 'Electronic traction control',
                    'description' => 'Electronic traction control',
                    'value' => 'standard',
                ],
                5 => [
                    'id' => 14801,
                    'name' => 'Central door locking',
                    'description' => 'Central door locking: Operated by remote Includes windows',
                    'value' => 'standard',
                ],
                6 => [
                    'id' => 14901,
                    'name' => 'Anti-theft protection',
                    'description' => 'Anti-theft protection with interior monitoring',
                    'value' => 'standard',
                ],
                7 => [
                    'id' => 16301,
                    'name' => 'Front airbag',
                    'description' => 'Drivers airbag, passengers airbag with de-activation switch',
                    'value' => 'standard',
                ],
                8 => [
                    'id' => 16401,
                    'name' => 'Side airbag',
                    'description' => 'Front side airbag',
                    'value' => 'standard',
                ],
                9 => [
                    'id' => 17501,
                    'name' => 'Front seat centre armrest',
                    'description' => 'Centre armrest between front seats',
                    'value' => 'standard',
                ],
                10 => [
                    'id' => 17601,
                    'name' => 'Rear seat centre armrest',
                    'description' => 'Centre armrest between rear seats',
                    'value' => 'standard',
                ],
                11 => [
                    'id' => 18501,
                    'name' => 'Power steering',
                    'description' => 'Type speed proportional power steering',
                    'value' => 'standard',
                ],
                12 => [
                    'id' => 19601,
                    'name' => 'Cup holders',
                    'description' => 'Cup holders for front seats and rear seats',
                    'value' => 'standard',
                ],
                13 => [
                    'id' => 21001,
                    'name' => 'Air conditioning',
                    'description' => 'Semi-automatic air conditioning',
                    'value' => 'standard',
                ],
                14 => [
                    'id' => 23301,
                    'name' => 'Electric windows',
                    'description' => 'Front and rear electric windows with one-touch on two windows',
                    'value' => 'standard',
                ],
                15 => [
                    'id' => 30701,
                    'name' => 'Stability control',
                    'description' => 'Stability controlsystem',
                    'value' => 'standard',
                ],
                16 => [
                    'id' => 32101,
                    'name' => 'Electronic brake distribution',
                    'description' => 'Electronic brake distribution',
                    'value' => 'standard',
                ],
                17 => [
                    'id' => 32201,
                    'name' => 'Brake assist system',
                    'description' => 'Brake assist system',
                    'value' => 'standard',
                ],
                18 => [
                    'id' => 35601,
                    'name' => 'Isofix preparation',
                    'description' => 'Isofix preparation',
                    'value' => 'standard',
                ],
                19 => [
                    'id' => 41001,
                    'name' => 'Disc player',
                    'description' => '1 disc CD autochanger in glove box',
                    'value' => 'standard',
                ],
                20 => [
                    'id' => 42201,
                    'name' => 'Knee airbags',
                    'description' => 'Knee airbags on driver side',
                    'value' => 'standard',
                ],
                21 => [
                    'id' => 44801,
                    'name' => 'Bluetooth',
                    'description' => 'Bluetooth includes phone connection',
                    'value' => 'standard',
                ],
                22 => [
                    'id' => 45001,
                    'name' => 'Start/Stop',
                    'description' => 'Engine start/stop',
                    'value' => 'standard',
                ],
                23 => [
                    'id' => 46901,
                    'name' => 'Connection to ext.entertainment devices',
                    'description' => 'Connections for iPod, USB and auxiliary audio devices',
                    'value' => 'standard',
                ],
            ],
            'standardSpecification' => [
                0 => [
                    'Interior' => [
                        0 => [
                            'id' => 701,
                            'name' => 'Seating',
                            'description' => 'Five seats configured 2+3',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 702,
                                    'name' => 'Seating capacity',
                                    'value' => 5,
                                ],
                                1 => [
                                    'id' => 703,
                                    'name' => 'seating configuration',
                                    'value' => '2+3',
                                ],
                            ],
                        ],
                        1 => [
                            'id' => 1101,
                            'name' => 'Speakers',
                            'description' => 'Eight speakers',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 1102,
                                    'name' => 'number of',
                                    'value' => 8,
                                ],
                            ],
                        ],
                        2 => [
                            'id' => 1301,
                            'name' => 'Audio player',
                            'description' => 'Audio system with digital media card reader and one disc autochanger ; radio receives AM/FM, digital and RDS',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 1302,
                                    'name' => 'radio',
                                    'value' => 'AM/FM',
                                ],
                                1 => [
                                    'id' => 1309,
                                    'name' => 'RDS',
                                    'value' => 'yes',
                                ],
                                2 => [
                                    'id' => 1315,
                                    'name' => 'Disc Autochanger',
                                    'value' => 'yes',
                                ],
                                3 => [
                                    'id' => 1323,
                                    'name' => 'disc autochanger capacity',
                                    'value' => 1,
                                ],
                                4 => [
                                    'id' => 1328,
                                    'name' => 'digital media card',
                                    'value' => 'yes',
                                ],
                                5 => [
                                    'id' => 1329,
                                    'name' => 'digital radio',
                                    'value' => 'yes',
                                ],
                            ],
                        ],
                        3 => [
                            'id' => 1401,
                            'name' => 'Remote audio controls',
                            'description' => 'Steering wheel mounted remote audio controls',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 1402,
                                    'name' => 'type',
                                    'value' => 'steering wheel mounted',
                                ],
                            ],
                        ],
                        4 => [
                            'id' => 1901,
                            'name' => '12v power outlet',
                            'description' => '12v power outlet located in cargo section and front section',
                            'value' => 'standard',
                        ],
                        5 => [
                            'id' => 4001,
                            'name' => 'Remote boot/hatch/rear door release',
                            'description' => 'Remote controlled remote boot/hatch/rear door release',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 4002,
                                    'name' => 'operation',
                                    'value' => 'remote control',
                                ],
                            ],
                        ],
                        6 => [
                            'id' => 5001,
                            'name' => 'Reading lights',
                            'description' => 'Front and rear reading lights',
                            'value' => 'standard',
                        ],
                        7 => [
                            'id' => 5301,
                            'name' => 'Load area light',
                            'description' => 'Load area light',
                            'value' => 'standard',
                        ],
                        8 => [
                            'id' => 5501,
                            'name' => 'Vanity mirror',
                            'description' => 'Illuminated driver and passenger vanity mirror',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 5503,
                                    'name' => 'illuminated',
                                    'location' => 'Driver',
                                    'value' => 'yes',
                                ],
                                1 => [
                                    'id' => 5503,
                                    'name' => 'illuminated',
                                    'value' => 'yes',
                                ],
                            ],
                        ],
                        9 => [
                            'id' => 9001,
                            'name' => 'Instrument panel',
                            'description' => 'Analogue dashboard',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 9002,
                                    'name' => 'type',
                                    'value' => 'analogue',
                                ],
                            ],
                        ],
                        10 => [
                            'id' => 9101,
                            'name' => 'Service interval indicator',
                            'description' => 'Service interval indicator',
                            'value' => 'standard',
                        ],
                        11 => [
                            'id' => 9501,
                            'name' => 'Tachometer',
                            'description' => 'Tachometer',
                            'value' => 'standard',
                        ],
                        12 => [
                            'id' => 10301,
                            'name' => 'External temperature',
                            'description' => 'External temperature',
                            'value' => 'standard',
                        ],
                        13 => [
                            'id' => 11801,
                            'name' => 'Computer',
                            'description' => 'Computer includes average speed, average fuel consumption, instantaneous fuel consumption and range for remaining fuel',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 11803,
                                    'name' => 'average speed',
                                    'value' => 'yes',
                                ],
                                1 => [
                                    'id' => 11804,
                                    'name' => 'average fuel consumption',
                                    'value' => 'yes',
                                ],
                                2 => [
                                    'id' => 11805,
                                    'name' => 'instantaneous fuel consumption',
                                    'value' => 'yes',
                                ],
                                3 => [
                                    'id' => 11806,
                                    'name' => 'range for remaining fuel',
                                    'value' => 'yes',
                                ],
                            ],
                        ],
                        14 => [
                            'id' => 11901,
                            'name' => 'Luxury trim',
                            'description' => 'Alloy trim on dashboard, alloy trim on doors, alloy trim on centre console and leather gearknob',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 11903,
                                    'name' => 'on gearknob',
                                    'value' => 'leather',
                                ],
                                1 => [
                                    'id' => 11905,
                                    'name' => 'on centre floor console',
                                    'value' => 'alloy',
                                ],
                                2 => [
                                    'id' => 11906,
                                    'name' => 'on doors',
                                    'value' => 'alloy',
                                ],
                                3 => [
                                    'id' => 11907,
                                    'name' => 'on dashboard',
                                    'value' => 'alloy',
                                ],
                            ],
                        ],
                        15 => [
                            'id' => 14701,
                            'name' => 'Load restraint',
                            'description' => 'Load restraint',
                            'value' => 'standard',
                        ],
                        16 => [
                            'id' => 14801,
                            'name' => 'Central door locking',
                            'description' => 'Central door locking: Operated by remote Includes windows',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 14802,
                                    'name' => 'operation',
                                    'value' => 'remote',
                                ],
                                1 => [
                                    'id' => 14807,
                                    'name' => 'includes electric windows',
                                    'value' => 'yes',
                                ],
                            ],
                        ],
                        17 => [
                            'id' => 17401,
                            'name' => 'Seat upholstery',
                            'description' => 'Cloth and cloth seat upholstery',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 17402,
                                    'name' => 'main seat material',
                                    'value' => 'cloth',
                                ],
                                1 => [
                                    'id' => 17403,
                                    'name' => 'additional seat material',
                                    'value' => 'cloth',
                                ],
                            ],
                        ],
                        18 => [
                            'id' => 17501,
                            'name' => 'Front seat centre armrest',
                            'description' => 'Centre armrest between front seats',
                            'value' => 'standard',
                        ],
                        19 => [
                            'id' => 17601,
                            'name' => 'Rear seat centre armrest',
                            'description' => 'Centre armrest between rear seats',
                            'value' => 'standard',
                        ],
                        20 => [
                            'id' => 17801,
                            'name' => 'Front seat',
                            'description' => 'Driver and passenger seat: individual type',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 17803,
                                    'name' => 'type',
                                    'location' => 'Driver',
                                    'value' => 'individual',
                                ],
                                1 => [
                                    'id' => 17803,
                                    'name' => 'type',
                                    'value' => 'individual',
                                ],
                                2 => [
                                    'id' => 17813,
                                    'name' => 'number of electrical adjustments',
                                    'location' => 'Driver',
                                    'value' => '-',
                                ],
                                3 => [
                                    'id' => 17813,
                                    'name' => 'number of electrical adjustments',
                                    'value' => '-',
                                ],
                                4 => [
                                    'id' => 17825,
                                    'name' => 'reclining type',
                                    'location' => 'Driver',
                                    'value' => 'manual',
                                ],
                                5 => [
                                    'id' => 17825,
                                    'name' => 'reclining type',
                                    'value' => 'manual',
                                ],
                                6 => [
                                    'id' => 17826,
                                    'name' => 'reach type',
                                    'location' => 'Driver',
                                    'value' => 'manual',
                                ],
                                7 => [
                                    'id' => 17826,
                                    'name' => 'reach type',
                                    'value' => 'manual',
                                ],
                                8 => [
                                    'id' => 17827,
                                    'name' => 'height type',
                                    'location' => 'Driver',
                                    'value' => 'manual',
                                ],
                                9 => [
                                    'id' => 17827,
                                    'name' => 'height type',
                                    'value' => 'manual',
                                ],
                                10 => [
                                    'id' => 17828,
                                    'name' => 'lumbar type',
                                    'location' => 'Driver',
                                    'value' => 'manual',
                                ],
                                11 => [
                                    'id' => 17828,
                                    'name' => 'lumbar type',
                                    'value' => 'manual',
                                ],
                            ],
                        ],
                        21 => [
                            'id' => 17901,
                            'name' => 'Rear seats',
                            'description' => 'Asymmetrically folding rear seats: bench type with space for three people; with zero electric adjustments and ski hatch',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 17903,
                                    'name' => 'type',
                                    'location' => 'Front',
                                    'value' => 'bench',
                                ],
                                1 => [
                                    'id' => 17910,
                                    'name' => 'number of electrical adjustments',
                                    'location' => 'Front',
                                    'value' => 0,
                                ],
                                2 => [
                                    'id' => 17912,
                                    'name' => 'folding',
                                    'location' => 'Front',
                                    'value' => 'asymmetrical',
                                ],
                                3 => [
                                    'id' => 17913,
                                    'name' => 'squab flip-up',
                                    'location' => 'Front',
                                    'value' => 'fixed',
                                ],
                                4 => [
                                    'id' => 17915,
                                    'name' => 'seating capacity',
                                    'location' => 'Front',
                                    'value' => 3,
                                ],
                                5 => [
                                    'id' => 17928,
                                    'name' => 'ski hatch',
                                    'location' => 'Front',
                                    'value' => 'yes',
                                ],
                            ],
                        ],
                        22 => [
                            'id' => 18401,
                            'name' => 'Steering wheel',
                            'description' => 'Leather covered multi-function steering wheel with tilt adjustment and telescopic adjustment',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 18402,
                                    'name' => 'type',
                                    'value' => 'leather covered',
                                ],
                                1 => [
                                    'id' => 18406,
                                    'name' => 'height adjustment',
                                    'value' => 'yes',
                                ],
                                2 => [
                                    'id' => 18407,
                                    'name' => 'telescopic adjustment',
                                    'value' => 'yes',
                                ],
                                3 => [
                                    'id' => 18410,
                                    'name' => 'multi-function',
                                    'value' => 'yes',
                                ],
                            ],
                        ],
                        23 => [
                            'id' => 18801,
                            'name' => 'Refrigerated storage compartment',
                            'description' => 'Refrigerated storage compartment in glove compartment cooled by air conditioning outlet',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 18803,
                                    'name' => 'type',
                                    'value' => 'ventilated',
                                ],
                            ],
                        ],
                        24 => [
                            'id' => 19101,
                            'name' => 'Below seat storage',
                            'description' => 'Storage compartment under driver seat and passenger seat',
                            'value' => 'standard',
                        ],
                        25 => [
                            'id' => 19601,
                            'name' => 'Cup holders',
                            'description' => 'Cup holders for front seats and rear seats',
                            'value' => 'standard',
                        ],
                        26 => [
                            'id' => 20801,
                            'name' => 'Ventilation system',
                            'description' => 'Ventilation system with air filter',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 20809,
                                    'name' => 'air filter',
                                    'value' => 'yes',
                                ],
                            ],
                        ],
                        27 => [
                            'id' => 21001,
                            'name' => 'Air conditioning',
                            'description' => 'Semi-automatic air conditioning',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 21009,
                                    'name' => 'air conditioning type',
                                    'value' => 'semi-auto',
                                ],
                            ],
                        ],
                        28 => [
                            'id' => 21501,
                            'name' => 'Rear view mirror',
                            'description' => 'Light sensitive rear view mirror',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 21503,
                                    'name' => 'auto-dimming',
                                    'value' => 'yes',
                                ],
                            ],
                        ],
                        29 => [
                            'id' => 23301,
                            'name' => 'Electric windows',
                            'description' => 'Front and rear electric windows with one-touch on two windows',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 23307,
                                    'name' => 'number of one touch',
                                    'location' => 'Front',
                                    'value' => 2,
                                ],
                                1 => [
                                    'id' => 23307,
                                    'name' => 'number of one touch',
                                    'location' => 'Rear',
                                    'value' => 2,
                                ],
                            ],
                        ],
                        30 => [
                            'id' => 26601,
                            'name' => 'Console',
                            'description' => 'Floor console, partial overhead console',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 26603,
                                    'name' => 'type',
                                    'value' => 'partial',
                                ],
                            ],
                        ],
                        31 => [
                            'id' => 41001,
                            'name' => 'Disc player',
                            'description' => '1 disc CD autochanger in glove box',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 41003,
                                    'name' => 'capacity',
                                    'value' => 1,
                                ],
                                1 => [
                                    'id' => 41004,
                                    'name' => 'media type',
                                    'value' => 'CD',
                                ],
                            ],
                        ],
                        32 => [
                            'id' => 41201,
                            'name' => 'Electronic hand brake',
                            'description' => 'Electronic hand brake',
                            'value' => 'standard',
                        ],
                        33 => [
                            'id' => 42301,
                            'name' => 'Storage box',
                            'description' => 'Fixed storage box in the boot',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 42303,
                                    'name' => 'type',
                                    'value' => 'fixed',
                                ],
                            ],
                        ],
                        34 => [
                            'id' => 43501,
                            'name' => 'Doors',
                            'description' => 'Front hinged drivers door, rear door on drivers side, front passenger door and rear door on passenger side',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 43503,
                                    'name' => 'type',
                                    'location' => 'Driver',
                                    'value' => 'front hinged',
                                ],
                                1 => [
                                    'id' => 43503,
                                    'name' => 'type',
                                    'value' => 'front hinged',
                                ],
                                2 => [
                                    'id' => 43503,
                                    'name' => 'type',
                                    'location' => 'Left',
                                    'value' => 'front hinged',
                                ],
                                3 => [
                                    'id' => 43503,
                                    'name' => 'type',
                                    'location' => 'Rear',
                                    'value' => 'front hinged',
                                ],
                            ],
                        ],
                        35 => [
                            'id' => 44801,
                            'name' => 'Bluetooth',
                            'description' => 'Bluetooth includes phone connection',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 44803,
                                    'name' => 'includes phone connection',
                                    'value' => 'yes',
                                ],
                            ],
                        ],
                        36 => [
                            'id' => 44901,
                            'name' => 'Rear door',
                            'description' => 'Lifting gate rear door',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 44902,
                                    'name' => 'type',
                                    'value' => 'lifting gate',
                                ],
                            ],
                        ],
                        37 => [
                            'id' => 45001,
                            'name' => 'Start/Stop',
                            'description' => 'Engine start/stop',
                            'value' => 'standard',
                        ],
                        38 => [
                            'id' => 46901,
                            'name' => 'Connection to ext.entertainment devices',
                            'description' => 'Connections for iPod, USB and auxiliary audio devices',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 46905,
                                    'name' => 'ipod front',
                                    'value' => 'yes',
                                ],
                                1 => [
                                    'id' => 46906,
                                    'name' => 'AUX jack front',
                                    'value' => 'yes',
                                ],
                                2 => [
                                    'id' => 46907,
                                    'name' => 'USB connection front',
                                    'value' => 'yes',
                                ],
                            ],
                        ],
                        39 => [
                            'id' => 50001,
                            'name' => 'Multi-function display screen',
                            'description' => '5.8 inch multi-function display screen',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 50002,
                                    'name' => 'screen size (in)',
                                    'value' => 5.8,
                                ],
                            ],
                        ],
                        40 => [
                            'id' => 54801,
                            'name' => 'Dynamic steering',
                            'description' => 'Dynamic steering',
                            'value' => 'standard',
                        ],
                    ],
                ],
                1 => [
                    'Exterior' => [
                        0 => [
                            'id' => 1201,
                            'name' => 'Aerial',
                            'description' => 'Element aerial',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 1202,
                                    'name' => 'type',
                                    'value' => 'element',
                                ],
                            ],
                        ],
                        1 => [
                            'id' => 3301,
                            'name' => 'Bumpers',
                            'description' => 'Painted front and rear bumpers',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 3305,
                                    'name' => 'colour',
                                    'location' => 'Front',
                                    'value' => 'painted',
                                ],
                                1 => [
                                    'id' => 3305,
                                    'name' => 'colour',
                                    'location' => 'Rear',
                                    'value' => 'painted',
                                ],
                            ],
                        ],
                        2 => [
                            'id' => 13401,
                            'name' => 'Day time running lights',
                            'description' => 'Day time running lights',
                            'value' => 'standard',
                        ],
                        3 => [
                            'id' => 14101,
                            'name' => 'Tyres',
                            'description' => 'Front and rear tyres: 16 inch diameter, 205mm wide, 55% profile, V rated ; Load index: 91 (official data)',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 14103,
                                    'name' => 'tyre width',
                                    'location' => 'Front',
                                    'value' => 205,
                                ],
                                1 => [
                                    'id' => 14103,
                                    'name' => 'tyre width',
                                    'location' => 'Rear',
                                    'value' => 205,
                                ],
                                2 => [
                                    'id' => 14104,
                                    'name' => 'tyre profile',
                                    'location' => 'Front',
                                    'value' => 55,
                                ],
                                3 => [
                                    'id' => 14104,
                                    'name' => 'tyre profile',
                                    'location' => 'Rear',
                                    'value' => 55,
                                ],
                                4 => [
                                    'id' => 14105,
                                    'name' => 'tyre speed rating',
                                    'location' => 'Front',
                                    'value' => 'V',
                                ],
                                5 => [
                                    'id' => 14105,
                                    'name' => 'tyre speed rating',
                                    'location' => 'Rear',
                                    'value' => 'V',
                                ],
                                6 => [
                                    'id' => 14109,
                                    'name' => 'tyre load index',
                                    'location' => 'Front',
                                    'value' => 91,
                                ],
                                7 => [
                                    'id' => 14109,
                                    'name' => 'tyre load index',
                                    'location' => 'Rear',
                                    'value' => 91,
                                ],
                                8 => [
                                    'id' => 14110,
                                    'name' => 'type',
                                    'location' => 'Front',
                                    'value' => 'conventional',
                                ],
                                9 => [
                                    'id' => 14110,
                                    'name' => 'type',
                                    'location' => 'Rear',
                                    'value' => 'conventional',
                                ],
                                10 => [
                                    'id' => 14114,
                                    'name' => 'official brochure tyre size',
                                    'location' => 'Front',
                                    'value' => 'yes',
                                ],
                                11 => [
                                    'id' => 14114,
                                    'name' => 'official brochure tyre size',
                                    'location' => 'Rear',
                                    'value' => 'yes',
                                ],
                                12 => [
                                    'id' => 14122,
                                    'name' => 'diameter',
                                    'location' => 'Front',
                                    'value' => 16,
                                ],
                                13 => [
                                    'id' => 14122,
                                    'name' => 'diameter',
                                    'location' => 'Rear',
                                    'value' => 16,
                                ],
                            ],
                        ],
                        4 => [
                            'id' => 15201,
                            'name' => 'Paint',
                            'description' => 'Gloss paint',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 15202,
                                    'name' => 'type',
                                    'value' => 'gloss',
                                ],
                            ],
                        ],
                        5 => [
                            'id' => 21601,
                            'name' => 'Door mirrors',
                            'description' => 'Body colour power door mirrors ; heated with integrated indicator lights',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 21603,
                                    'name' => 'type',
                                    'location' => 'Driver',
                                    'value' => 'electrically adjustable',
                                ],
                                1 => [
                                    'id' => 21603,
                                    'name' => 'type',
                                    'value' => 'electrically adjustable',
                                ],
                                2 => [
                                    'id' => 21605,
                                    'name' => 'heated',
                                    'location' => 'Driver',
                                    'value' => 'yes',
                                ],
                                3 => [
                                    'id' => 21605,
                                    'name' => 'heated',
                                    'value' => 'yes',
                                ],
                                4 => [
                                    'id' => 21607,
                                    'name' => 'colour',
                                    'location' => 'Driver',
                                    'value' => 'painted',
                                ],
                                5 => [
                                    'id' => 21607,
                                    'name' => 'colour',
                                    'value' => 'painted',
                                ],
                                6 => [
                                    'id' => 21611,
                                    'name' => 'indicator lights',
                                    'location' => 'Driver',
                                    'value' => 'yes',
                                ],
                                7 => [
                                    'id' => 21611,
                                    'name' => 'indicator lights',
                                    'value' => 'yes',
                                ],
                            ],
                        ],
                        6 => [
                            'id' => 22301,
                            'name' => 'Rear windscreen',
                            'description' => 'Rear windscreen with intermittent wiper',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 22302,
                                    'name' => 'type',
                                    'value' => 'fixed',
                                ],
                                1 => [
                                    'id' => 22306,
                                    'name' => 'wipers',
                                    'value' => 'intermittent',
                                ],
                            ],
                        ],
                        7 => [
                            'id' => 22501,
                            'name' => 'Tinted glass',
                            'description' => 'Tinted glass',
                            'value' => 'standard',
                        ],
                        8 => [
                            'id' => 22801,
                            'name' => 'Windscreen wipers',
                            'description' => 'Rain sensing windscreen wipers',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 22807,
                                    'name' => 'rain sensor',
                                    'value' => 'yes',
                                ],
                            ],
                        ],
                        9 => [
                            'id' => 23401,
                            'name' => 'Heated washers',
                            'description' => 'Heated washers',
                            'value' => 'standard',
                        ],
                        10 => [
                            'id' => 24401,
                            'name' => 'Wheels',
                            'description' => 'Front and rear wheels: 16 inch alloy rims ; width: 6.5 inches',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 24404,
                                    'name' => 'rim type',
                                    'location' => 'Front',
                                    'value' => 'alloy',
                                ],
                                1 => [
                                    'id' => 24404,
                                    'name' => 'rim type',
                                    'location' => 'Rear',
                                    'value' => 'alloy',
                                ],
                                2 => [
                                    'id' => 24405,
                                    'name' => 'rim diameter (in)',
                                    'location' => 'Front',
                                    'value' => 16,
                                ],
                                3 => [
                                    'id' => 24405,
                                    'name' => 'rim diameter (in)',
                                    'location' => 'Rear',
                                    'value' => 16,
                                ],
                                4 => [
                                    'id' => 24406,
                                    'name' => 'rim width (in)',
                                    'location' => 'Front',
                                    'value' => 6.5,
                                ],
                                5 => [
                                    'id' => 24406,
                                    'name' => 'rim width (in)',
                                    'location' => 'Rear',
                                    'value' => 6.5,
                                ],
                            ],
                        ],
                        11 => [
                            'id' => 24501,
                            'name' => 'Spare wheel',
                            'description' => 'Space saver steel spare wheel',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 24502,
                                    'name' => 'type',
                                    'value' => 'space saver',
                                ],
                                1 => [
                                    'id' => 24503,
                                    'name' => 'rim type',
                                    'value' => 'steel',
                                ],
                            ],
                        ],
                        12 => [
                            'id' => 24601,
                            'name' => 'Non-corrosive body',
                            'description' => 'Galvanised body',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 24602,
                                    'name' => 'type',
                                    'value' => 'galvanised',
                                ],
                            ],
                        ],
                        13 => [
                            'id' => 50201,
                            'name' => 'LED lights',
                            'description' => 'LED side indicator lights',
                            'value' => 'standard',
                        ],
                    ],
                ],
                2 => [
                    'Dimensions' => [
                        0 => [
                            'id' => 5801,
                            'name' => 'External dimensions',
                            'description' => 'External dimensions: 4,255mm long, 1,799mm wide, 1,452mm high, 142mm ground clearance, 2,637mm wheelbase, 1,549mm front track, 1,520mm rear track, 10,900mm turning circle (wall to wall), 167.5 inches long, 70.8 inches wide, 57.2 inches high, 5.6 inch ground clearance, 103.8 inch wheelbase, 61.0 inch front track and 59.8 inch rear track',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 5802,
                                    'name' => 'overall length (mm)',
                                    'value' => 4255,
                                ],
                                1 => [
                                    'id' => 5803,
                                    'name' => 'overall width (mm)',
                                    'value' => 1799,
                                ],
                                2 => [
                                    'id' => 5804,
                                    'name' => 'overall height (mm)',
                                    'value' => 1452,
                                ],
                                3 => [
                                    'id' => 5805,
                                    'name' => 'ground clearance unladen (mm)',
                                    'value' => 142,
                                ],
                                4 => [
                                    'id' => 5806,
                                    'name' => 'wheelbase (mm)',
                                    'value' => 2637,
                                ],
                                5 => [
                                    'id' => 5807,
                                    'name' => 'front track (mm)',
                                    'value' => 1549,
                                ],
                                6 => [
                                    'id' => 5808,
                                    'name' => 'rear track (mm)',
                                    'value' => 1520,
                                ],
                                7 => [
                                    'id' => 5810,
                                    'name' => 'wall to wall turning circle (mm)',
                                    'value' => 10900,
                                ],
                                8 => [
                                    'id' => 105802,
                                    'name' => 'overall length (in)',
                                    'value' => 167.5,
                                ],
                                9 => [
                                    'id' => 105803,
                                    'name' => 'overall width (in)',
                                    'value' => 70.8,
                                ],
                                10 => [
                                    'id' => 105804,
                                    'name' => 'overall height (in)',
                                    'value' => 57.2,
                                ],
                                11 => [
                                    'id' => 105805,
                                    'name' => 'ground clearance unladen (lbs)',
                                    'value' => 5.6,
                                ],
                                12 => [
                                    'id' => 105806,
                                    'name' => 'wheelbase (in)',
                                    'value' => 103.8,
                                ],
                                13 => [
                                    'id' => 105807,
                                    'name' => 'front track (in)',
                                    'value' => 61,
                                ],
                                14 => [
                                    'id' => 105808,
                                    'name' => 'rear track (in)',
                                    'value' => 59.8,
                                ],
                                15 => [
                                    'id' => 105810,
                                    'name' => 'wall to wall turning circle (ft)',
                                    'value' => 35.8,
                                ],
                            ],
                        ],
                        1 => [
                            'id' => 5901,
                            'name' => 'Internal dimensions',
                            'description' => 'Internal dimensions: front headroom (mm): 975, front headroom (inches): 38.4, rear headroom (mm): 967, rear headroom (inches): 38.1, front hip room (mm): 1,469, front hip room (inches): 57.8, rear hip room (mm): 1,440, rear hip room (inches): 56.7, front leg room (mm): 1,046, front leg room (inches): 41.2, rear leg room (mm): 903 and rear leg room (inches): 35.6',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 5902,
                                    'name' => 'headroom front (mm)',
                                    'value' => 975,
                                ],
                                1 => [
                                    'id' => 5903,
                                    'name' => 'headroom rear (mm)',
                                    'value' => 967,
                                ],
                                2 => [
                                    'id' => 5904,
                                    'name' => 'hip room front (mm)',
                                    'value' => 1469,
                                ],
                                3 => [
                                    'id' => 5905,
                                    'name' => 'hip room rear (mm)',
                                    'value' => 1440,
                                ],
                                4 => [
                                    'id' => 5906,
                                    'name' => 'leg room front (mm)',
                                    'value' => 1046,
                                ],
                                5 => [
                                    'id' => 5907,
                                    'name' => 'leg room rear (mm)',
                                    'value' => 903,
                                ],
                                6 => [
                                    'id' => 105902,
                                    'name' => 'headroom front (in)',
                                    'value' => 38.4,
                                ],
                                7 => [
                                    'id' => 105903,
                                    'name' => 'headroom rear (in)',
                                    'value' => 38.1,
                                ],
                                8 => [
                                    'id' => 105904,
                                    'name' => 'hip room front (in)',
                                    'value' => 57.8,
                                ],
                                9 => [
                                    'id' => 105905,
                                    'name' => 'hip room rear (in)',
                                    'value' => 56.7,
                                ],
                                10 => [
                                    'id' => 105906,
                                    'name' => 'leg room front (in)',
                                    'value' => 41.2,
                                ],
                                11 => [
                                    'id' => 105907,
                                    'name' => 'leg room rear (in)',
                                    'value' => 35.6,
                                ],
                            ],
                        ],
                        2 => [
                            'id' => 6001,
                            'name' => 'Load compartment capacity',
                            'description' => 'Load compartment capacity: 380 litres with rear seats up filled to lower edge of window, 1,270 litres with rear seats down filled to ceiling, 13.4 cu ft with rear seats up filled to lower edge of window and 44.8 cu ft with rear seats down filled to ceiling (using ISO measurement standards)',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 6002,
                                    'name' => 'rear seat up to lower window (l)',
                                    'value' => 380,
                                ],
                                1 => [
                                    'id' => 6004,
                                    'name' => 'rear seat down to roof (l)',
                                    'value' => 1270,
                                ],
                                2 => [
                                    'id' => 6008,
                                    'name' => 'measurement standard',
                                    'value' => 'ISO',
                                ],
                                3 => [
                                    'id' => 106002,
                                    'name' => 'rear seat up to lower window (cu ft)',
                                    'value' => 13.4,
                                ],
                                4 => [
                                    'id' => 106004,
                                    'name' => 'rear seat down to roof (cu ft)',
                                    'value' => 44.8,
                                ],
                            ],
                        ],
                        3 => [
                            'id' => 8901,
                            'name' => 'Fuel tank',
                            'description' => '50 litre, 11.0 UK gallon, 0.00 kg main fuel tank',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 8903,
                                    'name' => 'capacity (l)',
                                    'value' => 50,
                                ],
                                1 => [
                                    'id' => 8906,
                                    'name' => 'LPG/CNG capacity (Kg)',
                                    'value' => 0,
                                ],
                                2 => [
                                    'id' => 108903,
                                    'name' => 'capacity (gal)',
                                    'value' => 13.2,
                                ],
                                3 => [
                                    'id' => 108993,
                                    'name' => 'capacity (UK gallons)',
                                    'value' => 11,
                                ],
                            ],
                        ],
                        4 => [
                            'id' => 24101,
                            'name' => 'Weights',
                            'description' => 'Weights: gross vehicle weight rating (kg): 1,820, gross vehicle weight rating (lbs): 4,012, curb weight (kg): 1,295, curb weight (lbs): 2,855, gross trailer weight braked (kg): 1,800, gross trailer weight braked (lbs): 3,968, gross trailer weight unbraked (kg): 640, gross trailer weight unbraked (lbs): 1,411 and kerb weight includes driver: kerb weight includes driver',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 24102,
                                    'name' => 'gross vehicle weight (kg)',
                                    'value' => 1820,
                                ],
                                1 => [
                                    'id' => 24103,
                                    'name' => 'published kerb weight (kg)',
                                    'value' => 1295,
                                ],
                                2 => [
                                    'id' => 24105,
                                    'name' => 'gross trailer weight braked (kg)',
                                    'value' => 1800,
                                ],
                                3 => [
                                    'id' => 24106,
                                    'name' => 'gross trailer weight unbraked (kg)',
                                    'value' => 640,
                                ],
                                4 => [
                                    'id' => 24112,
                                    'name' => 'kerb weight includes driver',
                                    'value' => 'yes',
                                ],
                                5 => [
                                    'id' => 124102,
                                    'name' => 'gross vehicle weight (lbs)',
                                    'value' => 4012,
                                ],
                                6 => [
                                    'id' => 124103,
                                    'name' => 'published kerb weight (lbs)',
                                    'value' => 2855,
                                ],
                                7 => [
                                    'id' => 124105,
                                    'name' => 'gross trailer weight braked (lbs)',
                                    'value' => 3968,
                                ],
                                8 => [
                                    'id' => 124106,
                                    'name' => 'gross trailer weight unbraked (lbs)',
                                    'value' => 1411,
                                ],
                            ],
                        ],
                    ],
                ],
                3 => [
                    'Safety' => [
                        0 => [
                            'id' => 3101,
                            'name' => 'Disc brakes',
                            'description' => 'Four disc brakes including two ventilated discs',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 3102,
                                    'name' => 'number of',
                                    'value' => 4,
                                ],
                                1 => [
                                    'id' => 3103,
                                    'name' => 'number of ventilated discs',
                                    'value' => 2,
                                ],
                            ],
                        ],
                        1 => [
                            'id' => 3201,
                            'name' => 'ABS',
                            'description' => 'ABS',
                            'value' => 'standard',
                        ],
                        2 => [
                            'id' => 6901,
                            'name' => 'Electronic traction control',
                            'description' => 'Electronic traction control',
                            'value' => 'standard',
                        ],
                        3 => [
                            'id' => 12501,
                            'name' => 'Headlights',
                            'description' => 'Halogen headlights',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 12503,
                                    'name' => 'bulb type (low beam)',
                                    'value' => 'halogen',
                                ],
                            ],
                        ],
                        4 => [
                            'id' => 12601,
                            'name' => 'Headlight control',
                            'description' => 'Headlight controlsystems: dusk sensor and height adjustment',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 12604,
                                    'name' => 'dusk sensor',
                                    'value' => 'yes',
                                ],
                                1 => [
                                    'id' => 12608,
                                    'name' => 'height adjustment',
                                    'value' => 'manual',
                                ],
                            ],
                        ],
                        5 => [
                            'id' => 13801,
                            'name' => 'Side curtain airbag',
                            'description' => 'Front and rear side curtain airbags',
                            'value' => 'standard',
                        ],
                        6 => [
                            'id' => 16301,
                            'name' => 'Front airbag',
                            'description' => 'Drivers airbag, passengers airbag with de-activation switch',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 16305,
                                    'name' => 'occupant switch off',
                                    'value' => 'yes',
                                ],
                            ],
                        ],
                        7 => [
                            'id' => 16401,
                            'name' => 'Side airbag',
                            'description' => 'Front side airbag',
                            'value' => 'standard',
                        ],
                        8 => [
                            'id' => 16501,
                            'name' => 'Head restraints',
                            'description' => 'Two head restraints on front seats with height adjustment, three head restraints on rear seats with height adjustment',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 16504,
                                    'name' => 'height adjustable',
                                    'location' => 'Front',
                                    'value' => 'yes',
                                ],
                                1 => [
                                    'id' => 16504,
                                    'name' => 'height adjustable',
                                    'location' => 'Rear',
                                    'value' => 'yes',
                                ],
                                2 => [
                                    'id' => 16508,
                                    'name' => 'number',
                                    'location' => 'Front',
                                    'value' => 2,
                                ],
                                3 => [
                                    'id' => 16508,
                                    'name' => 'number',
                                    'location' => 'Rear',
                                    'value' => 3,
                                ],
                            ],
                        ],
                        9 => [
                            'id' => 16701,
                            'name' => 'Front seat belts',
                            'description' => 'Height adjustable seat belt with pre-tensioners on drivers seat and front passenger seat',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 16706,
                                    'name' => 'pre-tensioners',
                                    'location' => 'Driver',
                                    'value' => 'yes',
                                ],
                                1 => [
                                    'id' => 16706,
                                    'name' => 'pre-tensioners',
                                    'value' => 'yes',
                                ],
                                2 => [
                                    'id' => 16707,
                                    'name' => 'height adjustable',
                                    'location' => 'Driver',
                                    'value' => 'yes',
                                ],
                                3 => [
                                    'id' => 16707,
                                    'name' => 'height adjustable',
                                    'value' => 'yes',
                                ],
                            ],
                        ],
                        10 => [
                            'id' => 16801,
                            'name' => 'Rear seat belts',
                            'description' => 'Seat belt on driver side rear seat, seat belt on passenger side rear seat, 3-point seat belt on central rear seat',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 16803,
                                    'name' => 'type',
                                    'value' => '3-point',
                                ],
                            ],
                        ],
                        11 => [
                            'id' => 30701,
                            'name' => 'Stability control',
                            'description' => 'Stability controlsystem',
                            'value' => 'standard',
                        ],
                        12 => [
                            'id' => 32101,
                            'name' => 'Electronic brake distribution',
                            'description' => 'Electronic brake distribution',
                            'value' => 'standard',
                        ],
                        13 => [
                            'id' => 32201,
                            'name' => 'Brake assist system',
                            'description' => 'Brake assist system',
                            'value' => 'standard',
                        ],
                        14 => [
                            'id' => 33801,
                            'name' => 'Low tyre pressure indicator',
                            'description' => 'Low tyre pressure indicator',
                            'value' => 'standard',
                        ],
                        15 => [
                            'id' => 35601,
                            'name' => 'Isofix preparation',
                            'description' => 'Isofix preparation',
                            'value' => 'standard',
                        ],
                        16 => [
                            'id' => 42101,
                            'name' => 'Crash test results',
                            'description' => 'Euro NCAP crash test results: Overall: 5.00, Adult occupant: 94.00, Child occupant: 89.00, Pedestrian: 65.00, Safety assist: 71.00, Version tested: VW Golf 1.2 TSI Comfortline 5dr HA, Date tested: 28 Nov 2012',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 42117,
                                    'name' => 'date tested',
                                    'value' => 20121128,
                                ],
                                1 => [
                                    'id' => 42118,
                                    'name' => 'version tested',
                                    'value' => 'VW Golf 1.2 TSI Comfortline 5dr HA',
                                ],
                                2 => [
                                    'id' => 42119,
                                    'name' => 'overall',
                                    'value' => 5,
                                ],
                                3 => [
                                    'id' => 42120,
                                    'name' => 'adult occupant score',
                                    'value' => 94,
                                ],
                                4 => [
                                    'id' => 42121,
                                    'name' => 'child occupant score',
                                    'value' => 89,
                                ],
                                5 => [
                                    'id' => 42122,
                                    'name' => 'pedestrian score',
                                    'value' => 65,
                                ],
                                6 => [
                                    'id' => 42123,
                                    'name' => 'safety assist score',
                                    'value' => 71,
                                ],
                            ],
                        ],
                        17 => [
                            'id' => 42201,
                            'name' => 'Knee airbags',
                            'description' => 'Knee airbags on driver side',
                            'value' => 'standard',
                        ],
                        18 => [
                            'id' => 44001,
                            'name' => 'Automatic hazard lights',
                            'description' => 'Automatic hazard lights',
                            'value' => 'standard',
                        ],
                        19 => [
                            'id' => 44101,
                            'name' => 'Collision warning system',
                            'description' => 'Collision warning system which activates seat belts, activates brake lights and activates at low speed and includes brake assist and driver monitor',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 44102,
                                    'name' => 'activates seat belts',
                                    'value' => 'yes',
                                ],
                                1 => [
                                    'id' => 44103,
                                    'name' => 'activates brake lights',
                                    'value' => 'yes',
                                ],
                                2 => [
                                    'id' => 44105,
                                    'name' => 'driver monitor',
                                    'value' => 'yes',
                                ],
                                3 => [
                                    'id' => 44106,
                                    'name' => 'automatic braking',
                                    'value' => 'yes',
                                ],
                                4 => [
                                    'id' => 44107,
                                    'name' => 'brakes at low speed',
                                    'value' => 'front',
                                ],
                            ],
                        ],
                        20 => [
                            'id' => 46701,
                            'name' => 'Trailer stability program',
                            'description' => 'Trailer stability program',
                            'value' => 'standard',
                        ],
                    ],
                ],
                4 => [
                    'Security' => [
                        0 => [
                            'id' => 14901,
                            'name' => 'Anti-theft protection',
                            'description' => 'Anti-theft protection with interior monitoring',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 14905,
                                    'name' => 'interior monitoring',
                                    'value' => 'yes',
                                ],
                            ],
                        ],
                    ],
                ],
                5 => [
                    'Technical' => [
                        0 => [
                            'id' => 4501,
                            'name' => 'Cruise control',
                            'description' => 'Adaptive cruise control',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 4503,
                                    'name' => 'adaptive cruise control',
                                    'value' => 'yes',
                                ],
                            ],
                        ],
                        1 => [
                            'id' => 6501,
                            'name' => 'Drive',
                            'description' => 'Front-wheel drive',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 6502,
                                    'name' => 'Driven wheels',
                                    'value' => 'front',
                                ],
                            ],
                        ],
                        2 => [
                            'id' => 6801,
                            'name' => 'Limited slip differential',
                            'description' => 'Front electronic limited slip differential',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 6803,
                                    'name' => 'type',
                                    'location' => 'Front',
                                    'value' => 'electronic',
                                ],
                            ],
                        ],
                        3 => [
                            'id' => 7401,
                            'name' => 'Engine',
                            'description' => '1,598 cc 1.6 litres in-line 4 engine with 79.5 mm bore, 80.5 mm stroke, 16.2 compression ratio, double overhead cam and four valves per cylinder',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 7402,
                                    'name' => 'cc',
                                    'value' => 1598,
                                ],
                                1 => [
                                    'id' => 7403,
                                    'name' => 'Litres',
                                    'value' => 1.6,
                                ],
                                2 => [
                                    'id' => 7404,
                                    'name' => 'bore',
                                    'value' => 79.5,
                                ],
                                3 => [
                                    'id' => 7405,
                                    'name' => 'stroke',
                                    'value' => 80.5,
                                ],
                                4 => [
                                    'id' => 7406,
                                    'name' => 'compression ratio',
                                    'value' => 16.2,
                                ],
                                5 => [
                                    'id' => 7407,
                                    'name' => 'number of cylinders',
                                    'value' => 4,
                                ],
                                6 => [
                                    'id' => 7408,
                                    'name' => 'configuration',
                                    'value' => 'in-line',
                                ],
                                7 => [
                                    'id' => 7414,
                                    'name' => 'valve gear type',
                                    'value' => 'double overhead cam',
                                ],
                                8 => [
                                    'id' => 7417,
                                    'name' => 'number of valves per cylinder',
                                    'value' => 4,
                                ],
                            ],
                        ],
                        4 => [
                            'id' => 7501,
                            'name' => 'Compressor',
                            'description' => 'Turbo-charger',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 7502,
                                    'name' => 'Compressor',
                                    'value' => 'turbo',
                                ],
                            ],
                        ],
                        5 => [
                            'id' => 7601,
                            'name' => 'Emission control level',
                            'description' => 'Emissions: EU5 compliant, 99 g/km CO2 (max), 0.14500 g/km carbon monoxide, 0.139 g/km HC+NOx, 0.118 g/km NOx, 0.000 g/km particulate matter and 72 dB noise level',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 7602,
                                    'name' => 'standard met',
                                    'value' => 'EU5',
                                ],
                                1 => [
                                    'id' => 7603,
                                    'name' => 'CO2 level - g/km combined',
                                    'value' => 99,
                                ],
                                2 => [
                                    'id' => 7608,
                                    'name' => 'CO (g/km)',
                                    'value' => 0.145,
                                ],
                                3 => [
                                    'id' => 7609,
                                    'name' => 'emission particles (g/km)',
                                    'value' => 0,
                                ],
                                4 => [
                                    'id' => 7611,
                                    'name' => 'HC + NOx (g/km)',
                                    'value' => 0.139,
                                ],
                                5 => [
                                    'id' => 7612,
                                    'name' => 'NOx level (g/km)',
                                    'value' => 0.118,
                                ],
                                6 => [
                                    'id' => 7617,
                                    'name' => 'noise level (dB)',
                                    'value' => 72,
                                ],
                            ],
                        ],
                        6 => [
                            'id' => 8501,
                            'name' => 'Fuel system',
                            'description' => 'Fuel system: common rail diesel',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 8502,
                                    'name' => 'injection/carburation',
                                    'value' => 'diesel common rail',
                                ],
                            ],
                        ],
                        7 => [
                            'id' => 8701,
                            'name' => 'Fuel',
                            'description' => 'Primary fuel type: diesel',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 8702,
                                    'name' => 'Fuel type',
                                    'value' => 'diesel',
                                ],
                                1 => [
                                    'id' => 8708,
                                    'name' => 'generic primary fuel type',
                                    'value' => 'diesel',
                                ],
                            ],
                        ],
                        8 => [
                            'id' => 13501,
                            'name' => 'Performance',
                            'description' => 'Performance: maximum speed 119 mph, 192 km/h acceleration 10.7 seconds (0-100 km/h)',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 13502,
                                    'name' => 'maximum speed (km/h)',
                                    'value' => 192,
                                ],
                                1 => [
                                    'id' => 13503,
                                    'name' => 'acceleration 0-62mph (s)',
                                    'value' => 10.7,
                                ],
                                2 => [
                                    'id' => 113502,
                                    'name' => 'maximum speed (mph)',
                                    'value' => 119,
                                ],
                            ],
                        ],
                        9 => [
                            'id' => 15301,
                            'name' => 'Power',
                            'description' => 'Power: (from primary fuel) 77 kW , 105 HP EEC @ 3,000 rpm; , 250 Nm @ 1,500 rpm',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 15302,
                                    'name' => 'measurement standard',
                                    'value' => 'EEC',
                                ],
                                1 => [
                                    'id' => 15303,
                                    'name' => 'Maximum power kW',
                                    'value' => 77,
                                ],
                                2 => [
                                    'id' => 15304,
                                    'name' => 'Maximum power hp/PS',
                                    'value' => 105,
                                ],
                                3 => [
                                    'id' => 15305,
                                    'name' => 'rpm for maximum power (low)',
                                    'value' => 3000,
                                ],
                                4 => [
                                    'id' => 15307,
                                    'name' => 'maximum torque Nm',
                                    'value' => 250,
                                ],
                                5 => [
                                    'id' => 15308,
                                    'name' => 'rpm for maximum torque (low)',
                                    'value' => 1500,
                                ],
                                6 => [
                                    'id' => 15324,
                                    'name' => 'primary or secondary fuel power figure',
                                    'value' => 'primary',
                                ],
                            ],
                        ],
                        10 => [
                            'id' => 18501,
                            'name' => 'Power steering',
                            'description' => 'Type speed proportional power steering',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 18502,
                                    'name' => 'vehicle speed proportional',
                                    'value' => 'yes',
                                ],
                                1 => [
                                    'id' => 18503,
                                    'name' => 'type',
                                    'value' => 'electric',
                                ],
                            ],
                        ],
                        11 => [
                            'id' => 20001,
                            'name' => 'Suspension',
                            'description' => 'Independent strut front suspension with anti-roll bar and coil springs, semi-independent torsion beam rear suspension with coil springs',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 20002,
                                    'name' => 'type',
                                    'location' => 'Front',
                                    'value' => 'strut',
                                ],
                                1 => [
                                    'id' => 20002,
                                    'name' => 'type',
                                    'location' => 'Rear',
                                    'value' => 'torsion beam',
                                ],
                                2 => [
                                    'id' => 20003,
                                    'name' => 'anti-roll bar',
                                    'location' => 'Front',
                                    'value' => 'yes',
                                ],
                                3 => [
                                    'id' => 20005,
                                    'name' => 'wheel dependence',
                                    'location' => 'Front',
                                    'value' => 'independent',
                                ],
                                4 => [
                                    'id' => 20005,
                                    'name' => 'wheel dependence',
                                    'location' => 'Rear',
                                    'value' => 'semi-independent',
                                ],
                                5 => [
                                    'id' => 20006,
                                    'name' => 'spring type',
                                    'location' => 'Front',
                                    'value' => 'coil',
                                ],
                                6 => [
                                    'id' => 20006,
                                    'name' => 'spring type',
                                    'location' => 'Rear',
                                    'value' => 'coil',
                                ],
                            ],
                        ],
                        12 => [
                            'id' => 20601,
                            'name' => 'Transmission',
                            'description' => 'Manual five-speed transmission with shift lever on floor manual',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 20602,
                                    'name' => 'Transmission type',
                                    'value' => 'manual',
                                ],
                                1 => [
                                    'id' => 20603,
                                    'name' => 'number of speeds',
                                    'value' => 5,
                                ],
                                2 => [
                                    'id' => 20610,
                                    'name' => 'gearchange location',
                                    'value' => 'floor',
                                ],
                                3 => [
                                    'id' => 20624,
                                    'name' => 'transmission description',
                                    'value' => 'manual',
                                ],
                            ],
                        ],
                        13 => [
                            'id' => 33501,
                            'name' => 'Traffic light coding',
                            'description' => 'Traffic light coding: efficiency label A',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 33502,
                                    'name' => 'efficiency label',
                                    'value' => 'A',
                                ],
                            ],
                        ],
                        14 => [
                            'id' => 38701,
                            'name' => 'Intermediate service',
                            'description' => 'Intermediate servicerequired after variable distance/period variable distance/period, 16,093 km, 10,000 miles, 12 months',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 38702,
                                    'name' => 'distance (km)',
                                    'value' => 16093,
                                ],
                                1 => [
                                    'id' => 38703,
                                    'name' => 'period (mths)',
                                    'value' => 12,
                                ],
                                2 => [
                                    'id' => 38704,
                                    'name' => 'variable distance/period',
                                    'value' => 'yes',
                                ],
                                3 => [
                                    'id' => 38705,
                                    'name' => 'distance (miles)',
                                    'value' => 10000,
                                ],
                            ],
                        ],
                        15 => [
                            'id' => 38801,
                            'name' => 'Main service',
                            'description' => 'Main servicerequired after variable distance/period, 32,187 km, 20,000 miles, 24 months',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 38802,
                                    'name' => 'distance (km)',
                                    'value' => 32187,
                                ],
                                1 => [
                                    'id' => 38803,
                                    'name' => 'period (mths)',
                                    'value' => 24,
                                ],
                                2 => [
                                    'id' => 38804,
                                    'name' => 'variable distance/period',
                                    'value' => 'yes',
                                ],
                                3 => [
                                    'id' => 38805,
                                    'name' => 'distance (miles)',
                                    'value' => 20000,
                                ],
                            ],
                        ],
                        16 => [
                            'id' => 41601,
                            'name' => 'Particle filter system',
                            'description' => 'Particle filter system',
                            'value' => 'standard',
                        ],
                        17 => [
                            'id' => 42001,
                            'name' => 'Fuel consumption',
                            'description' => 'Fuel consumption: 4.6 l/100km (urban), 3.3 l/100km (country/highway), 3.8 l/100km (combined), 51 mpg (urban US), 71 mpg (country/highway US), 62 mpg (combined US), 21.7 km/l (urban), 30.3 km/l (country/highway), 26.3 km/l (combined), 61 mpg (urban UK), 86 mpg (country/highway UK), 74 mpg (combined UK) ; Range 1,316 km (combined), 818 miles (combined) ; Standard: ECE 99/100',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 42003,
                                    'name' => 'urban (l/100km)',
                                    'value' => 4.6,
                                ],
                                1 => [
                                    'id' => 42004,
                                    'name' => 'country/highway (l/100km)',
                                    'value' => 3.3,
                                ],
                                2 => [
                                    'id' => 42005,
                                    'name' => 'combined (l/100km)',
                                    'value' => 3.8,
                                ],
                                3 => [
                                    'id' => 42006,
                                    'name' => 'urban (km/l)',
                                    'value' => 21.7,
                                ],
                                4 => [
                                    'id' => 42007,
                                    'name' => 'country/highway (km/l)',
                                    'value' => 30.3,
                                ],
                                5 => [
                                    'id' => 42008,
                                    'name' => 'combined (km/l)',
                                    'value' => 26.3,
                                ],
                                6 => [
                                    'id' => 42015,
                                    'name' => 'combined vehicle range (km)',
                                    'value' => 1316,
                                ],
                                7 => [
                                    'id' => 42016,
                                    'name' => 'combined vehicle range (miles)',
                                    'value' => 818,
                                ],
                                8 => [
                                    'id' => 142003,
                                    'name' => 'urban (mpg)',
                                    'value' => 51,
                                ],
                                9 => [
                                    'id' => 142004,
                                    'name' => 'country/highway (mpg)',
                                    'value' => 71,
                                ],
                                10 => [
                                    'id' => 142005,
                                    'name' => 'combined (mpg)',
                                    'value' => 62,
                                ],
                                11 => [
                                    'id' => 142006,
                                    'name' => 'urban (miles per UK gallon)',
                                    'value' => 61,
                                ],
                                12 => [
                                    'id' => 142007,
                                    'name' => 'country/highway (miles per UK gallon)',
                                    'value' => 86,
                                ],
                                13 => [
                                    'id' => 142008,
                                    'name' => 'combined (miles per UK gallon)',
                                    'value' => 74,
                                ],
                            ],
                        ],
                        18 => [
                            'id' => 43201,
                            'name' => 'Hill holder',
                            'description' => 'Hill holder',
                            'value' => 'standard',
                        ],
                        19 => [
                            'id' => 48601,
                            'name' => 'Powertrain type',
                            'description' => 'Powertrain: internal combustion engine',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 48602,
                                    'name' => 'type',
                                    'value' => 'combustion',
                                ],
                            ],
                        ],
                        20 => [
                            'id' => 51301,
                            'name' => 'Energy recuperation',
                            'description' => 'Regenerative brakes',
                            'value' => 'standard',
                        ],
                    ],
                ],
                6 => [
                    'Others' => [
                        0 => [
                            'id' => 174,
                            'name' => 'Global segment',
                            'description' => 'Global segment',
                            'value' => 'Lower Medium',
                        ],
                        1 => [
                            'id' => 176,
                            'name' => 'Regional segment',
                            'description' => 'Regional segment',
                            'value' => 'C1 - lower medium -',
                        ],
                        2 => [
                            'id' => 401,
                            'name' => 'Trim',
                            'description' => 'SE trim level - JATO classification: M1',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 402,
                                    'name' => 'Trim level',
                                    'value' => 'SE',
                                ],
                                1 => [
                                    'id' => 404,
                                    'name' => 'local trim level',
                                    'value' => 'SE',
                                ],
                                2 => [
                                    'id' => 405,
                                    'name' => 'trim classification',
                                    'value' => 'M1',
                                ],
                            ],
                        ],
                        3 => [
                            'id' => 601,
                            'name' => 'Body style',
                            'description' => 'Five door hatchback ; Platform code: MQB, Right hand drive',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 602,
                                    'name' => 'Number of doors',
                                    'value' => 5,
                                ],
                                1 => [
                                    'id' => 603,
                                    'name' => 'Body type',
                                    'value' => 'hatchback',
                                ],
                                2 => [
                                    'id' => 605,
                                    'name' => 'local number of doors',
                                    'value' => 5,
                                ],
                                3 => [
                                    'id' => 606,
                                    'name' => 'local body type',
                                    'value' => 'hatchback',
                                ],
                                4 => [
                                    'id' => 608,
                                    'name' => 'wheelbase type',
                                    'value' => 'short',
                                ],
                                5 => [
                                    'id' => 609,
                                    'name' => 'Driver location',
                                    'value' => 'RHD',
                                ],
                                6 => [
                                    'id' => 610,
                                    'name' => 'platform code',
                                    'value' => 'MQB',
                                ],
                            ],
                        ],
                        4 => [
                            'id' => 2501,
                            'name' => 'Insurance',
                            'description' => 'Insurance: 13E-T1',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 2502,
                                    'name' => 'description',
                                    'value' => '13E-T1',
                                ],
                            ],
                        ],
                        5 => [
                            'id' => 3501,
                            'name' => 'Charges',
                            'description' => 'Charges: VAT [20%] = 3,350.83, VED (Band A) = 0.00, FRF = 55.00, Showroom Tax = 0.00',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 3510,
                                    'name' => 'national tax 1 name',
                                    'value' => 'VAT [20%]',
                                ],
                                1 => [
                                    'id' => 3512,
                                    'name' => 'national tax 2 name',
                                    'value' => 'VED (Band A)',
                                ],
                                2 => [
                                    'id' => 3514,
                                    'name' => 'national tax 3 name',
                                    'value' => 'FRF',
                                ],
                                3 => [
                                    'id' => 3516,
                                    'name' => 'national tax 1 amount',
                                    'value' => 3350.83,
                                ],
                                4 => [
                                    'id' => 3517,
                                    'name' => 'national tax 2 amount',
                                    'value' => 0,
                                ],
                                5 => [
                                    'id' => 3518,
                                    'name' => 'national tax 3 amount',
                                    'value' => 55,
                                ],
                                6 => [
                                    'id' => 3519,
                                    'name' => 'national tax 4 name',
                                    'value' => 'Showroom Tax',
                                ],
                                7 => [
                                    'id' => 3520,
                                    'name' => 'national tax 4 amount',
                                    'value' => 0,
                                ],
                            ],
                        ],
                        6 => [
                            'id' => 3601,
                            'name' => 'Delivery charges',
                            'description' => 'Fixed delivery charges: 550.00, 458.33 (excluding VAT) fuel included 0 litres includes number plates',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 3602,
                                    'name' => 'type',
                                    'value' => 'fixed',
                                ],
                                1 => [
                                    'id' => 3603,
                                    'name' => 'amount',
                                    'value' => 550,
                                ],
                                2 => [
                                    'id' => 3604,
                                    'name' => 'amount excluding VAT',
                                    'value' => 458.33,
                                ],
                                3 => [
                                    'id' => 3605,
                                    'name' => 'includes fuel (l)',
                                    'value' => 0,
                                ],
                                4 => [
                                    'id' => 3606,
                                    'name' => 'includes number plates',
                                    'value' => 'yes',
                                ],
                            ],
                        ],
                        7 => [
                            'id' => 14601,
                            'name' => 'Cargo area cover/rear parcel shelf',
                            'description' => 'Rigid cargo area cover/rear parcel shelf',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 14602,
                                    'name' => 'type',
                                    'value' => 'rigid',
                                ],
                            ],
                        ],
                        8 => [
                            'id' => 23501,
                            'name' => 'Warranty whole vehicle - Total',
                            'description' => 'Vehicle warranty: 36 months, 60,000 miles, 96,560 km',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 23502,
                                    'name' => 'duration (months)',
                                    'value' => 36,
                                ],
                                1 => [
                                    'id' => 23503,
                                    'name' => 'distance (km)',
                                    'value' => 96560,
                                ],
                                2 => [
                                    'id' => 123503,
                                    'name' => 'distance (miles)',
                                    'value' => 60000,
                                ],
                            ],
                        ],
                        9 => [
                            'id' => 23601,
                            'name' => 'Warranty powertrain - Total',
                            'description' => 'Powertrain warranty: 36 months, 60,000 miles, 96,560 km',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 23602,
                                    'name' => 'duration (months)',
                                    'value' => 36,
                                ],
                                1 => [
                                    'id' => 23603,
                                    'name' => 'distance (km)',
                                    'value' => 96560,
                                ],
                                2 => [
                                    'id' => 123603,
                                    'name' => 'distance (miles)',
                                    'value' => 60000,
                                ],
                            ],
                        ],
                        10 => [
                            'id' => 23701,
                            'name' => 'Warranty anti-corrosion',
                            'description' => 'Anticorrosion warranty: 144 months, Unlimited miles, Unlimited km',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 23702,
                                    'name' => 'duration (months)',
                                    'value' => 144,
                                ],
                                1 => [
                                    'id' => 23703,
                                    'name' => 'distance (km)',
                                    'value' => 999999,
                                ],
                                2 => [
                                    'id' => 123703,
                                    'name' => 'distance (miles)',
                                    'value' => 999999,
                                ],
                            ],
                        ],
                        11 => [
                            'id' => 23801,
                            'name' => 'Warranty paint',
                            'description' => 'Paint warranty: 36 months, Unlimited miles, Unlimited km',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 23802,
                                    'name' => 'duration (months)',
                                    'value' => 36,
                                ],
                                1 => [
                                    'id' => 23803,
                                    'name' => 'distance (km)',
                                    'value' => 999999,
                                ],
                                2 => [
                                    'id' => 123803,
                                    'name' => 'distance (miles)',
                                    'value' => 999999,
                                ],
                            ],
                        ],
                        12 => [
                            'id' => 24001,
                            'name' => 'Warranty roadside assistance',
                            'description' => 'Road-side assistance warranty: 12 months, Unlimited miles, Unlimited km',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 24002,
                                    'name' => 'duration (months)',
                                    'value' => 12,
                                ],
                                1 => [
                                    'id' => 24003,
                                    'name' => 'distance (km)',
                                    'value' => 999999,
                                ],
                                2 => [
                                    'id' => 124003,
                                    'name' => 'distance (miles)',
                                    'value' => 999999,
                                ],
                            ],
                        ],
                        13 => [
                            'id' => 42901,
                            'name' => 'Date introduced',
                            'description' => 'Date introduced: Model introduced 18 Oct 2012 and Version introduced 18 Oct 2012',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 42908,
                                    'name' => 'Version introduced',
                                    'value' => 20121018,
                                ],
                                1 => [
                                    'id' => 42909,
                                    'name' => 'Model introduced',
                                    'value' => 20121018,
                                ],
                            ],
                        ],
                        14 => [
                            'id' => 48401,
                            'name' => 'Data completeness',
                            'description' => 'Data completeness: up to date pricing data, up to date data for key leasing items, specifications data from official documentation and physical inspection, up to date colour & trim data, up to date option pricing data, up to date option content data and incomplete incentives data',
                            'value' => 'standard',
                            'items' => [
                                0 => [
                                    'id' => 48402,
                                    'name' => 'colour & trim status',
                                    'value' => 'up to date',
                                ],
                                1 => [
                                    'id' => 48403,
                                    'name' => 'key leasing items',
                                    'value' => 'up to date',
                                ],
                                2 => [
                                    'id' => 48404,
                                    'name' => 'option content status',
                                    'value' => 'up to date',
                                ],
                                3 => [
                                    'id' => 48405,
                                    'name' => 'option prices status',
                                    'value' => 'up to date',
                                ],
                                4 => [
                                    'id' => 48406,
                                    'name' => 'prices status',
                                    'value' => 'up to date',
                                ],
                                5 => [
                                    'id' => 48407,
                                    'name' => 'specs status',
                                    'value' => 'documentation + physical inspection',
                                ],
                                6 => [
                                    'id' => 48408,
                                    'name' => 'API Codes',
                                    'value' => 'incomplete',
                                ],
                            ],
                        ],
                    ],
                ],
            ],
            'serviceVersion' => 'V2',
            '_links' => [
                'self' => [
                    'href' => 'https://staging.motorspecs.com/specs/standard/1',
                ],
            ],
        ];
    }

    /**
     * https://documenter.getpostman.com/view/313931/SzKWuxGg?version=latest#81e07663-52b3-4c04-982d-3b2723b94fc2
     *
     * @return array
     */
    protected function valuationDataAT()
    {
        return [
            'vehicleId' => NULL,
            'registration' => 'HY63CBO',
            'currentMiles' => 0,
            'autotraderValuation' => [
                'ownershipCondition' => 'Used',
                'registration' => 'HY63CBO',
                'vin' => 'WVWZZZAUZEW080398',
                'make' => 'Volkswagen',
                'model' => 'Golf',
                'generation' => 'Hatchback (2012 - 2017) MK7',
                'derivative' => '1.6 TDI SE Hatchback 5dr Diesel Manual (s/s) (99 g/km, 104 bhp)',
                'derivativeId' => 'fe5b2a1d31a4d6dbb731fbee401215de',
                'vehicleType' => 'Car',
                'trim' => 'SE',
                'bodyType' => 'Hatchback',
                'fuelType' => 'Diesel',
                'cabType' => NULL,
                'transmissionType' => 'Manual',
                'drivetrain' => 'Front Wheel Drive',
                'seats' => 5,
                'doors' => 5,
                'co2' => 99,
                'engineSizeLitres' => 1,
                'engineCc' => 1598,
                'bhp' => 104,
                'emissionClass' => '0',
                'insuranceGroup' => '11',
                'insuranceSecurityCode' => 'E',
                'firstRegistrationDate' => '2013-09-14',
                'colour' => 'White',
                'engineNumber' => 'CLHA346534',
                'fuelDelivery' => 'Direct Injection',
                'gears' => 5,
                'driveType' => '4X2',
                'valuationMileage' => 0,
                'valuation' => [
                    'trade' => 7242,
                    'retail' => 9975,
                    'partExchange' => 7050,
                    'private' => 8390,
                ],
            ],
            '_links' => [
                'self' => [
                    'href' => 'https://staging.motorspecs.com/valuation-autotrader/value',
                ],
            ],
        ];
    }
    /**
     * https://documenter.getpostman.com/view/313931/SzKWuxGg?version=latest#e350877c-a274-45da-b35e-c318369ceaa2
     *
     * @return array
     */
    protected function valuationDataVIP()
    {
        return [
            'vehicleId' => NULL,
            'registration' => 'HY63CBO',
            'currentMiles' => 80000,
            'priceWhenNew' => 21190,
            'vipValuation' => [
                'forecourtExcellent' => 6810,
                'forecourtGood' => 6370,
                'forecourtPoor' => 5950,
                'privateExcellent' => 6370,
                'privateGood' => 5940,
                'privatePoor' => 5550,
                'tradeinExcellent' => 5860,
                'tradeinGood' => 5460,
                'tradeinPoor' => 5090,
                'auctionExcellent' => 5580,
                'auctionGood' => 5200,
                'auctionPoor' => 4850,
            ],
            'serviceVersion' => 'V2',
            '_links' => [
                'self' => [
                    'href' => 'https://staging.motorspecs.com/valuation-vip/value',
                ],
            ],
        ];
    }

    protected function reportSearchData()
    {
        return [
            'data' => [
                [
                    'report_id' => 'REPORT123'
                ]
            ]
        ];
    }

    protected function reportData()
    {
        return [
            'data' => [
                'issues' => [
                    'finance' => [
                        'flag' => true,
                        'message' => 'Finance agreement in place',
                    ],
                    'mileage' => [
                        'flag' => false,
                        'message' => 'No mileage discrepancies',
                    ],
                    'atRisk' => [
                        'flag' => false,
                        'message' => 'No risk marker recorded',
                    ],
                    'writeOff' => [
                        'flag' => false,
                        'message' => 'No write-off recorded',
                    ],
                    'conditionAlerts' => [
                        'flag' => false,
                        'message' => 'No condition alerts',
                    ],
                    'scrapped' => [
                        'flag' => false,
                        'message' => 'Not scrapped, no VIC recorded',
                    ],
                    'stolen' => [
                        'flag' => false,
                        'message' => 'No stolen record',
                    ],
                    'keepers' => [
                        'flag' => false,
                        'message' => 'No  issues recorded.',
                    ],
                    'plateChange' => [
                        'flag' => false,
                        'message' => 'No  issues recorded.',
                    ],
                    'colourChanges' => [
                        'flag' => false,
                        'message' => 'No  issues recorded.',
                    ],
                    'motHistory' => [
                        'flag' => false,
                        'message' => 'No  issues recorded.',
                    ],
                    'taxAndSorn' => [
                        'flag' => false,
                        'message' => 'No  issues recorded.',
                    ],
                    'originAndUse' => [
                        'flag' => false,
                        'message' => 'No  issues recorded.',
                    ],
                ],
                'certificate_pdf' => NULL,
            ]
        ];
    }
}
