<?php

namespace Mtc\MotorCheck\Services;

use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Http;
use Mtc\MotorCheck\Exceptions\UnauthorizedRequestException;

class MotorSpecsApi
{
    public function __construct(protected array $config = [])
    {
        //
    }

    /**
     * Retrieve full specs details for vehicle
     * This lists vehicle Identity (make model, details) as well as
     * standard features (enriched data, non-essential features) that are on the vehicle
     *
     * @param string $vrm
     * @param int $mileage
     * @return array
     * @throws \JsonException
     */
    public function getSpecs(string $vrm, int $mileage = 0): array
    {
        return [
            'identity' => $this->getIdentitySpecs($vrm, $mileage),
            'standard' => $this->getStandardSpecs($vrm),
        ];
    }

    /**
     * Retrieve vehicle valuation based on AutoTrader API in MotorSpecs
     *
     * @param string $vrm
     * @param int $mileage
     * @return array
     * @throws \Illuminate\Http\Client\RequestException
     * @throws \JsonException
     */
    public function getAutoTraderValuation(string $vrm, int $mileage = 0): array
    {
        $param_json = json_encode([
            'registration' => $vrm,
            'currentMiles' => $mileage,
        ], JSON_THROW_ON_ERROR);

        // Needs to be run before valuation even though we don't need the data
        $this->getIdentitySpecs($vrm, $mileage);

        return Http::withToken($this->getAccessToken())
            ->withHeaders($this->headers('vnd.valuation-auto-trader.v2+json'))
            ->withBody($param_json, 'application/vnd.valuation-auto-trader.v2+json')
            ->post($this->endpoint('valuation-autotrader/value'))
            ->throw()
            ->json();
    }

    /**
     * Retrieve vehicle valuation based on VIP API in MotorSpecs
     *
     * @param string $vrm
     * @param int $mileage
     * @return array
     * @throws \Illuminate\Http\Client\RequestException
     * @throws \JsonException
     */
    public function getVipValuation(string $vrm, int $mileage = 0): array
    {
        $param_json = json_encode([
            'registration' => $vrm,
            'currentMiles' => (int)$mileage,
        ], JSON_THROW_ON_ERROR);

        // Needs to be run before valuation even though we don't need the data
        $this->getIdentitySpecs($vrm, $mileage);

        return Http::withToken($this->getAccessToken())
            ->withHeaders($this->headers('vnd.valuation-vip.v2+json'))
            ->withBody($param_json, 'application/vnd.valuation-vip.v2+json')
            ->post($this->endpoint('valuation-vip/value'))
            ->throw()
            ->json();
    }

    /**
     * Retrieve vehicle Identity details (critical information) from MotorSpecs API
     *
     * @param $vrm
     * @param $mileage
     * @return array
     * @throws \Illuminate\Http\Client\RequestException
     * @throws \JsonException
     */
    protected function getIdentitySpecs($vrm, $mileage): array
    {
        $param_json = json_encode([
            'registration' => $vrm,
            'currentMiles' => (int)$mileage,
        ], JSON_THROW_ON_ERROR);

        return Http::withToken($this->getAccessToken())
            ->withHeaders($this->headers('vnd.identity-specs.v2+json'))
            ->withBody($param_json, 'application/vnd.identity-specs.v2+json')
            ->post($this->endpoint('identity-specs/lookup'))
            ->throw()
            ->json();
    }

    /**
     * Retrieve vehicle Standard Specs based on AutoTrader API in MotorSpecs
     * This lists features car has like alloy wheels and speaker count
     *
     * @param $vrm
     * @return array
     * @throws \Illuminate\Http\Client\RequestException
     * @throws \JsonException
     */
    protected function getStandardSpecs($vrm): array
    {
        $param_json = json_encode([
            'registration' => $vrm,
        ], JSON_THROW_ON_ERROR);

        return Http::withToken($this->getAccessToken())
            ->withHeaders($this->headers('vnd.specs.v2+json'))
            ->withBody($param_json, 'application/vnd.specs.v2+json')
            ->post($this->endpoint('specs/standard'))
            ->throw()
            ->json();
    }

    protected function getAccessToken()
    {
        return Cache::remember('motor-specs-api-token', now()->addDay(), function () {
            return $this->login();
        });
    }

    protected function login(): string
    {
        $param_json = json_encode([
            'grant_type' => 'client_credentials',
            'client_id' => $this->config['username'],
            'client_secret' => $this->config['password'],
            'login' => true,
        ], JSON_THROW_ON_ERROR);

        $result = Http::withBody($param_json, 'application/json')
            ->post($this->endpoint('oauth'));

        if ($result->failed()) {
            throw new UnauthorizedRequestException('Failed to authorize');
        }

        return $result->json('access_token');
    }

    protected function endpoint(string $path): string
    {
        return $this->config['live']
            ? 'https://api.motorspecs.com/' . $path
            : 'https://staging.motorspecs.com/' . $path;
    }

    protected function headers(string $content_type): array
    {
        return [
            'Content-Type: application/' . $content_type,
            'Accept: application/' . $content_type,
        ];
    }
}
