<?php

namespace Mtc\MotorCheck\Maps;

use Illuminate\Support\Str;
use Mtc\AutomotiveSpecs\VehicleSpecification;
use Mtc\MotorCheck\Contracts\MotorSpecsToVehicleSpecificationContract;

class MotorSpecsToVehicleSpecification implements MotorSpecsToVehicleSpecificationContract
{
    protected array $data;

    public function __construct(protected VehicleSpecification $spec)
    {
    }

    public function map(array $spec_data): VehicleSpecification
    {
        $this->data = $spec_data;
        $this->setBasicValues();
        $this->setFeatureList();
        return $this->spec;
    }

    protected function setBasicValues(): void
    {
        $this->spec->setValuesFromArray([
            'UUID' => $this->data['identity']['vehicleId'],
            'registration_number' => $this->data['identity']['vehicle']['combined']['registration'],
            'make' => $this->make(),
            'model' => $this->model(),
            'derivative' => $this->derivative(),
            'model_year' => $this->modelYear(),
            'registration_date' => $this->registrationDate(),
            'engine_size' => $this->engineSize(),
            'mpg' => $this->mpg(),
            'co2' => $this->co2(),
            'body_style' => $this->bodyStyle(),
            'colour' => $this->colour(),
            'transmission' => $this->transmission(),
            'fuel_type' => $this->fuelType(),
            'drivetrain' => $this->drivetrain(),
            'odometer_value' => $this->odometerValue(),
            'odometer_metric' => $this->odometerMetric(),
        ]);
    }

    protected function setFeatureList(): void
    {
        $feature_list = collect($this->data['standard']['standardSpecification'])
            ->map(function ($category_group) {
                return collect($category_group)
                    ->map(function ($category_items, $category) {
                        $category_items = collect($category_items)
                            ->map(function ($category_item) use ($category) {
                                return [
                                    'name' => $category_item['name'],
                                    'value' => $category_item['description'],
                                    'type' => $category
                                ];
                            });
                        return $category_items;
                    });
            })
            ->flatten(2);
        $this->spec->setFeatureList($feature_list->toArray());
    }

    protected function make(): string
    {
        return Str::title($this->data['identity']['specsVehicle']['make'] ?? '');
    }

    protected function model(): string
    {
        return Str::title($this->data['identity']['specsVehicle']['model'] ?? '');
    }

    protected function modelYear(): int
    {
        return $this->data['identity']['specsVehicle']['modelYear'] ?? 0;
    }

    protected function derivative(): string
    {
        return $this->data['identity']['specsVehicle']['version'] ?? '';
    }

    protected function registrationDate(): string
    {
        return $this->data['identity']['vehicle']['combined']['regDate'] ?? '';
    }

    protected function engineSize(): int
    {
        return (int)$this->data['identity']['specsVehicle']['engineCC'];
    }

    protected function mpg(): int
    {
        return (int)$this->data['identity']['vehicle']['mvris']['combinedMpg'];
    }

    protected function co2(): int
    {
        return (int)($this->data['identity']['vehicle']['dvla']['co2'] ?? 0);
    }

    protected function bodyStyle(): string
    {
        return Str::title($this->data['identity']['specsVehicle']['body'] ?? '');
    }

    protected function colour(): string
    {
        return Str::title($this->data['identity']['vehicle']['combined']['colour'] ?? '');
    }

    protected function transmission(): string
    {
        return Str::title($this->data['identity']['specsVehicle']['transmission'] ?? '');
    }

    protected function fuelType(): string
    {
        return Str::title($this->data['identity']['specsVehicle']['fuel'] ?? '');
    }

    protected function drivetrain(): string
    {
        return $this->data['identity']['vehicle']['mvris']['driveType'] ?? '';
    }

    protected function odometerValue(): string
    {
        return (int)$this->data['identity']['vehicle']['combined']['currentMiles'];
    }

    protected function odometerMetric(): string
    {
        return 'mi';
    }
}
