<?php

namespace Mtc\MercuryDataModels;

use App\TierHelper;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Mtc\ContentManager\Contracts\MediaUse;
use Mtc\ContentManager\Http\Resources\MediaEntryResource;
use Mtc\ContentManager\Traits\HasMedia;
use Mtc\MercuryDataModels\Casts\SpecifiedTypeCast;
use Mtc\MercuryDataModels\Events\SettingUpdated;
use Mtc\MercuryDataModels\Factories\SettingFactory;
use OwenIt\Auditing\Contracts\Auditable;

class Setting extends Model implements Auditable
{
    use \OwenIt\Auditing\Auditable;
    use HasFactory;
    use HasMedia;

    protected $dispatchesEvents = [
        'updated' => SettingUpdated::class,
    ];

    protected $fillable = [
        'tab',
        'section',
        'group',
        'name',
        'config_key',
        'type',
        'choices',
        'source',
        'value',
        'value_text',
        'order',
        'min_tier',
        'validation_rules',
        'superadmin',
        'feature',
        'description',
        'data'
    ];

    protected $casts = [
        'value' => SpecifiedTypeCast::class,
        'choices' => 'array',
        'validation_rules' => 'array',
        'data' => 'array'
    ];

    protected $hidden = [
        'order',
        'validation_rules',
        'id',
        'created_at',
        'updated_at',
        'media_uses',
    ];

    /**
     * Model factory
     *
     * @return SettingFactory
     */
    protected static function newFactory()
    {
        return SettingFactory::new();
    }

    protected static function boot()
    {
        parent::boot();
        self::saved(fn(self $setting) => $setting->checkIfCurrencyUpdate());
    }

    /**
     * Check if this setting value is available on current tier
     *
     * @param string $tier
     * @return bool
     */
    public function availableOnTier(string $tier): bool
    {
        if (is_null($this->min_tier)) {
            return true;
        }

        return TierHelper::isAllowed($tier, $this->min_tier);
    }

    public function hasFeature(?Collection $features): bool
    {
        if (is_null($this->feature)) {
            return true;
        }

        // Not available on this feature
        if (str_starts_with($this->feature, '!')) {
            return !($features && $features->search(ltrim($this->feature, '!')) !== false);
        }

        return $features && $features->search($this->feature) !== false;
    }


    public function setDynamicValues(string $tier)
    {
        if ($this->type === 'media') {
            $this->media = $this->mediaUses->pluck('media_id');
            $this->media_use = $this->mediaUses
                ->map(fn(MediaUse $mediaUse) => new MediaEntryResource($mediaUse));
        }
    }

    private function checkIfCurrencyUpdate()
    {
        if ($this->config_key === 'app-details-currency') {
            tenant()->currency = $this->value;
            tenant()->save();
        }
    }
}
