<?php

namespace Mtc\MercuryDataModels\Traits;

use Carbon\Carbon;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\MorphMany;
use Illuminate\Support\Facades\DB;
use Mtc\MercuryDataModels\ResourceView;

trait HasResourceViews
{
    /**
     * Relationship with resource views
     */
    public function resourceViews(): MorphMany
    {
        return $this->morphMany(ResourceView::class, 'viewable');
    }

    /**
     * Get total views count (all time)
     */
    public function getTotalViewsAttribute(): int
    {
        return (int) $this->resourceViews()->sum('hits');
    }

    /**
     * Get trending views count (last N days, default 30)
     */
    public function getTrendingViewsAttribute(): int
    {
        return $this->getViewsForDays(30);
    }

    /**
     * Get views for a specific number of days
     */
    public function getViewsForDays(int $days): int
    {
        return (int) $this->resourceViews()
            ->where('date', '>=', Carbon::now()->subDays($days))
            ->sum('hits');
    }

    /**
     * Scope to add total view count as a subquery
     */
    public function scopeWithTotalViewCount(Builder $query): void
    {
        $query->addSelect([
            'total_view_count' => ResourceView::query()
                ->select(DB::raw('COALESCE(SUM(hits), 0)'))
                ->whereColumn('viewable_id', $this->getTable() . '.id')
                ->where('viewable_type', $this->getMorphClass())
        ]);
    }

    /**
     * Scope to add trending view count as a subquery (last N days)
     */
    public function scopeWithTrendingViewCount(Builder $query, int $days = 30): void
    {
        $query->addSelect([
            'trending_view_count' => ResourceView::query()
                ->select(DB::raw('COALESCE(SUM(hits), 0)'))
                ->whereColumn('viewable_id', $this->getTable() . '.id')
                ->where('viewable_type', $this->getMorphClass())
                ->where('date', '>=', Carbon::now()->subDays($days))
        ]);
    }

    /**
     * Scope to order by popularity (total views)
     */
    public function scopeOrderByPopular(Builder $query, string $direction = 'desc'): void
    {
        $query->withTotalViewCount()->orderBy('total_view_count', $direction);
    }

    /**
     * Scope to order by trending (views in last N days)
     */
    public function scopeOrderByTrending(Builder $query, int $days = 30, string $direction = 'desc'): void
    {
        $query->withTrendingViewCount($days)->orderBy('trending_view_count', $direction);
    }
}
