<?php

namespace Mtc\MercuryDataModels\Services;

use App\Facades\Settings;
use Carbon\Carbon;
use Illuminate\Support\Facades\Http;
use Illuminate\Support\Facades\Log;
use Mtc\MercuryDataModels\ApiNotification;
use Mtc\MercuryDataModels\Services\Config\KeyloopReservationConfig;
use Mtc\MercuryDataModels\Vehicle;

class KeyloopApi
{
    private const ENDPOINT_VEHICLE_DETAILS = '/v1/inventory-vehicles/{vehicleId}';
    private const ENDPOINT_RESERVE_VEHICLE = '/v1/inventory-vehicles/{vehicleId}/reserve';

    /**
     * @param KeyloopReservationConfig $config
     */
    public function __construct(protected readonly KeyloopReservationConfig $config)
    {
        //
    }

    /**
     * @param Vehicle $vehicle
     * @return array|mixed|string[]
     */
    public function getVehicleDetails(Vehicle $vehicle)
    {
        $url = $this->endpoint(self::ENDPOINT_VEHICLE_DETAILS, $vehicle);
        $response = Http::withHeaders($this->getHeaders())
            ->get($url);

        ApiNotification::query()->create([
            'provider' => 'keyloop-stock-api',
            'processed' => $response->successful(),
            'reference' => $vehicle->vrm_condensed,
            'data' => [
                'url' => $url,
                'status_code' => $response->status(),
                'response' => $response->body(),
            ],
        ]);

        if ($response->successful()) {
            return $response->json();
        }

        Log::warning('failed to retrieve vehicle data', [
            'url' => $url,
        ]);

        return [
            'error' => 'failed to return any results'
        ];
    }

    /**
     * @param Vehicle $vehicle
     * @param bool $is_reserved
     * @return void
     */
    public function setVehicleReserved(Vehicle $vehicle)
    {
        $url = $this->endpoint(self::ENDPOINT_RESERVE_VEHICLE, $vehicle);
        $response = Http::withHeaders($this->getHeaders())
            ->post($url, [
                'reservation' => [
                    'reservationDate' => $this->getReservationDateTime(),
                    'reservedUntilDate' => null,
                    'reservedBy' => $this->getReservedBy(),
                ],
                'metadata' => [
                    'auditData' => [
                        'userId' => $this->config->userId(),
                        'userName' => $this->config->userName(),
                    ],
                ],
            ]);

        ApiNotification::query()->create([
            'provider' => 'keyloop-stock-api',
            'processed' => $response->successful(),
            'reference' => $vehicle->vrm_condensed,
            'data' => [
                'url' => $url,
                'status_code' => $response->status(),
                'response' => $response->body(),
            ],
        ]);

        if ($response->successful()) {
            return $response->json();
        }

        Log::warning('Failed to reserve vehicle in Keyloop', [
            'url' => $url,
            'result' => $response->body(),
        ]);
    }

    /**
     * @return array|null
     */
    protected function getReservedBy(): ?array
    {
        return [
            'salesPersonId' => Settings::get('keyloop-reservations-salesperson-id') ?: null,
            'name' => Settings::get('keyloop-reservations-salesperson-name') ?: null,
        ];
    }

    /**
     * Return current datetime in UTC, formated as Y-m-d H:i:s.v
     * e.g. 2021-07-16T00:00:00.000Z
     *
     * @return string
     */
    protected function getReservationDateTime(): string
    {
        $date_time = Carbon::now()->format('Y-m-d H:i:s.v');

        $utc_datetime = Carbon::createFromFormat(
            'Y-m-d H:i:s.v',
            $date_time,
            Settings::get('app-timezone')
        )->setTimezone('UTC');

        $utc_date_string = $utc_datetime->format('Y-m-d H:i:s.v');

        return str_replace(' ', 'T', $utc_date_string) . 'Z';
    }

    /**
     * @return string[]
     */
    protected function getHeaders(): array
    {
        return [
            'Authorization' => 'Bearer ' . $this->accessToken(),
        ];
    }

    /**
     * @return string|null
     */
    protected function accessToken(): ?string
    {
        $response = Http::withHeaders($this->getTokenHeaders())
            ->post($this->tokenEndpoint(), null);

        return $response->successful()
            ? $response->json('access_token') ?? null
            : null;
    }

    /**
     * @return string[]
     */
    protected function getTokenHeaders()
    {
        return [
            'Content-Type' => 'application/json',
            'Authorization' => 'Basic '
                . base64_encode($this->config->clientId() . ':' . $this->config->clientSecret()),
        ];
    }

    /**
     * @return string
     */
    protected function tokenEndpoint(): string
    {
        // Note that production and test mode token URLs seem to be the same
        return (app()->isProduction())
            ? 'https://api.eu.keyloop.io/oauth/client_credential/accesstoken?grant_type=client_credentials'
            : 'https://api.eu.keyloop.io/oauth/client_credential/accesstoken?grant_type=client_credentials';
    }

    /**
     * @param string $path
     * @param Vehicle $vehicle
     * @return string
     */
    protected function endpoint(string $path, Vehicle $vehicle): string
    {
        // Note that production and test mode base URLs seem to be the same
        $url = app()->isProduction()
            ? 'https://api.eu.keyloop.io/{enterpriseId}/{storeId}'
            : 'https://api.eu.keyloop.io/{enterpriseId}/{storeId}';

        $url = $url . $path;
        $url = str_replace('{enterpriseId}', $this->config->enterpriseId(), $url);
        $url = str_replace('{storeId}', $vehicle->dealership?->data['keyloop-dealer-id'] ?? '', $url);
        $url = str_replace(
            '{vehicleId}',
            $vehicle->attributeValues()
                ->where('slug', 'keyloop-id')
                ->first()->value ?? '',
            $url
        );

        return $url;
    }
}
