<?php

namespace Mtc\MercuryDataModels;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Support\Str;
use Mtc\MercuryDataModels\Factories\PropertyCategoryFieldFactory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use OwenIt\Auditing\Contracts\Auditable;

class PropertyCategoryField extends Model implements Auditable
{
    use \OwenIt\Auditing\Auditable;
    use HasFactory;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'property_category_fields';

    /**
     * Mass assign columns
     *
     * @var string[]
     */
    protected $fillable = [
        'category_id',
        'name',
        'slug',
        'type',
        'active',
        'order',
        'data',
        'validation',
        'choices',
    ];

    protected $casts = [
        'active' => 'boolean',
        'data' => 'array',
        'choices' => 'array',
    ];

    protected static function boot()
    {
        parent::boot();
        self::creating(fn(self $field) => $field->ensureSlug());
        self::deleted(fn(self $field) => $field->values()->delete());
    }

    protected static function newFactory()
    {
        return PropertyCategoryFieldFactory::new();
    }

    public function category(): BelongsTo
    {
        return  $this->belongsTo(PropertyCategory::class, 'category_id');
    }


    public function values(): HasMany
    {
        return  $this->hasMany(PropertyAttributeValue::class, 'field_id');
    }

    public function scopeActive(Builder $query): void
    {
        $query->where('active', 1);
    }

    private function ensureSlug()
    {
        if (empty($this->attributes['slug'])) {
            $slug = Str::slug($this->attributes['name']);
            $this->attributes['slug'] = $slug;

            $loop = 1;

            while (
                self::query()
                ->where('category_id', $this->category_id)
                ->where('slug', $this->slug)
                ->exists()
            ) {
                $this->attributes['slug'] = $slug . '-' . ($loop++);
            }
        }
    }
}
