<?php

namespace Mtc\MercuryDataModels;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Relations\BelongsTo;
use Illuminate\Support\Str;
use Mtc\ContentManager\Contracts\Menu as MenuModel;
use Mtc\MercuryDataModels\Contracts\SearchableModel;
use Mtc\MercuryDataModels\Tools\UiUrlGenerator;
use OwenIt\Auditing\Contracts\Auditable;

class Menu extends MenuModel implements SearchableModel, Auditable
{
    use \OwenIt\Auditing\Auditable;

    protected $fillable = [
        'title',
        'slug',
        'description',
        'first_child_element_id',
        'data',
        'meta',
        'is_enabled',
        'franchise_id',
    ];

    /**
     * Search name text
     *
     * @return string
     */
    public function getSearchNameAttribute(): string
    {
        return $this->title;
    }

    /**
     * Search excerpt text
     *
     * @return string
     */
    public function getSearchExcerptAttribute(): string
    {
        return '';
    }

    /**
     * Search result icon
     *
     * @return string
     */
    public function getSearchIcon(): string
    {
        return 'code-branch';
    }

    /**
     * Route to viewing a vehicle as a part of search response
     *
     * @return string
     */
    public function getSearchResultRoute(): string
    {
        return UiUrlGenerator::make('manage-content/menus/edit/' . $this->id, [], false);
    }

    public function franchise(): BelongsTo
    {
        return $this->belongsTo(Franchise::class, 'franchise_id');
    }

    public function scopeSetFilters(Builder $query, array|string $filters)
    {
        if (is_string($filters) && !empty($filters)) {
            $filters = [ $filters ];
        }

        collect($filters)
            ->filter(fn ($filter) => method_exists($this, 'scope' . ucfirst(Str::camel($filter))))
            ->map(fn ($filter) => 'scope' . ucfirst(Str::camel($filter)))
            ->each(fn ($scopeMethod) => $this->{$scopeMethod}($query));

        $notFound = collect($filters)
            ->reject(fn ($filter) => method_exists($this, 'scope' . ucfirst(Str::camel($filter))))
            ->filter();
        if ($notFound->count()) {
            $query->whereHas('franchise', fn($query) => $query->whereIn('slug', $notFound));
        }
    }
}
