<?php

namespace Mtc\MercuryDataModels\Jobs;

use Carbon\Carbon;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Facades\Mail;
use Mtc\MercuryDataModels\ApiNotification;
use Mtc\MercuryDataModels\ResourceView;
use Mtc\MercuryDataModels\Vehicle;

class VehicleComparisonMailRequest implements ShouldQueue
{
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    /**
     * Create a new job instance.
     */
    public function __construct(
        private readonly string $email,
        private readonly array $vehicle_slugs,
        private readonly string $comparison_url,
    ) {
        //
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        $success = false;
        $message = '';

        try {
            $this->trackComparisonRequests();
            $this->sendMail();
            $success = true;
            $message = 'success';
        } catch (\Exception $exception) {
            $message = get_class($exception) . ': ' . $exception->getMessage();
        } finally {
            ApiNotification::query()->create([
                'provider' => 'comparison_email_request',
                'processed' => $success,
                'data' => [
                    'slugs' => $this->vehicle_slugs,
                    'recipient' => $this->email,
                    'url' => $this->comparison_url,
                    'message' => $message,
                ],
            ]);
        }
    }

    private function trackComparisonRequests(): void
    {
        $vehicle_ids = collect($this->vehicle_slugs)->map(fn ($slug) =>
            Vehicle::query()->where('slug', $slug)->first()?->id ?? null)
            ->filter();

        $today = Carbon::today()->toDateString();

        foreach ($vehicle_ids as $vehicleId) {
            $view = ResourceView::query()
                ->where('viewable_type', 'vehicle')
                ->where('viewable_id', $vehicleId)
                ->whereDate('date', $today)
                ->first();

            if ($view) {
                ResourceView::query()
                    ->where('id', $view->id)
                    ->increment('compare_email_requests');
            } else {
                ResourceView::query()->create([
                    'viewable_type' => 'vehicle',
                    'viewable_id' => $vehicleId,
                    'date' => $today,
                    'compare_email_requests' => 1,
                ]);
            }
        }
    }

    private function sendMail(): void
    {
        $class = config('crm.vehicle_comparison_email');
        Mail::to($this->email)->send(new $class($this->comparison_url));
    }
}
