<?php

namespace Mtc\MercuryDataModels\Jobs;

use App\Master\Models\ApiToken;
use Carbon\Carbon;
use Illuminate\Bus\Queueable;
use Illuminate\Contracts\Queue\ShouldQueue;
use Illuminate\Foundation\Bus\Dispatchable;
use Illuminate\Queue\InteractsWithQueue;
use Illuminate\Queue\SerializesModels;
use Illuminate\Support\Str;
use Mtc\MercuryDataModels\ApiUsage;

class TrackApiRequest implements ShouldQueue
{
    use Dispatchable;
    use InteractsWithQueue;
    use Queueable;
    use SerializesModels;

    /**
     * Create a new job instance.
     */
    public function __construct(
        private readonly ?string $tenant_id,
        private readonly ?string $uri,
        private readonly ?array $input,
        private readonly ?string $method,
        private readonly ?int $status_code,
        private readonly ?float $request_length,
        private readonly ?ApiToken $api_token = null,
    ) {
        //
    }

    /**
     * Execute the job.
     */
    public function handle(): void
    {
        $usage = ApiUsage::query()->updateOrCreate([
            'tenant_id' => $this->tenant_id,
            'endpoint' => Str::limit($this->uri, 191, ''),
            'method' => $this->method,
            'time_window' => Carbon::now()->format('Y-m-d H'),
            'api_user_id' => $this->api_token?->api_user_id,
            'api_token_id' => $this->api_token?->type === 'persistent' ? $this->api_token?->id : null,
        ]);
        if ($this->shouldAddSnapshot($usage)) {
            $usage->snapshots()->create([
                'input' => $this->input,
                'request_time' => $this->request_length,
            ]);
            if ($usage->snapshots()->count() > 5) {
                $usage->snapshots()->orderBy('request_time')->first()->delete();
            }
        }
        $this->setResponseTimeDetails($usage);
    }

    private function shouldAddSnapshot($usage): bool
    {
        return $this->method === 'POST'
            && !empty($this->input)
            && $this->request_length > config('app.slow_request_time')
            && $usage->snapshots()->min('request_time') < $this->request_length;
    }

    private function setResponseTimeDetails($usage): void
    {
        $hits = ($usage->hits ?? 0) + 1;
        $avg = $usage->avg + (($this->request_length - $usage->avg) / $hits);

        $updateData = [
            'hits' => $hits,
            'avg' => $avg,
            'min' => ($usage->min ?? 60) > $this->request_length ? $this->request_length : $usage->min,
            'max' => $usage->max < $this->request_length ? $this->request_length : $usage->max,
        ];

        // Track slow requests (above 3s and above 10s)
        if ($this->request_length >= 3) {
            $updateData['slow_3s'] = ($usage->slow_3s ?? 0) + 1;
        }
        if ($this->request_length >= 10) {
            $updateData['slow_10s'] = ($usage->slow_10s ?? 0) + 1;
        }

        $usage->update($updateData);
    }
}
