<?php

namespace Mtc\MercuryDataModels;

use Illuminate\Database\Eloquent\Relations\BelongsToMany;
use Illuminate\Database\Eloquent\Relations\HasMany;
use Illuminate\Database\Eloquent\Relations\MorphOne;
use Mtc\Crm\Traits\ModelSortAndFilter;
use Mtc\Filter\FilterIndex;
use Mtc\MercuryDataModels\Factories\FuelTypeFactory;
use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Mtc\MercuryDataModels\Traits\EnsuresSlug;
use Mtc\MercuryDataModels\Traits\OrderByName;

class FuelType extends Model
{
    use HasFactory;
    use ModelSortAndFilter;
    use EnsuresSlug;

    /**
     * Table name
     *
     * @var string
     */
    protected $table = 'fuel_types';

    /**
     * Mass assign columns
     *
     * @var string[]
     */
    protected $fillable = [
        'id',
        'name',
        'slug',
        'ev_data',
        'mpg_data',
        'autotrader_id',
    ];

    /**
     * Columns shown in JSON response by default
     *
     * @var string[]
     */
    protected $visible = [
        'id',
        'name',
        'slug',
        'ev_data',
        'mpg_data',
    ];

    protected static function boot()
    {
        parent::boot();

        self::addGlobalScope(fn($query) => $query->orderBy('order')->orderBy('name'));

        self::saving(function (self $fuelType) {
            $fuelType->ensureSlug();
        });
    }

    /**
     * Model factory
     *
     * @return FuelTypeFactory
     */
    protected static function newFactory()
    {
        return FuelTypeFactory::new();
    }

    public function filterIndex(): MorphOne
    {
        return $this->morphOne(FilterIndex::class, 'filter');
    }

    /**
     * Relationship with vehicles
     *
     * @return HasMany
     */
    public function vehicles(): HasMany
    {
        return $this->hasMany(Vehicle::class, 'fuel_type_id');
    }

    /**
     * Relationship with offers
     *
     * @return HasMany
     */
    public function offers(): HasMany
    {
        return $this->hasMany(VehicleOffer::class, 'fuel_type_id');
    }

    /**
     * Relationship with new cars
     *
     * @return BelongsToMany
     */
    public function newCars(): BelongsToMany
    {
        return $this->belongsToMany(NewCar::class, 'new_car_fuel_types', 'fuel_type_id', 'new_car_id');
    }

    public function leaseVehicles(): HasMany
    {
        return $this->hasMany(KeyloopLeaseVehicleVariant::class, 'fuel_type_id');
    }

    protected function getSlugAttribute(): string
    {
        if (empty($this->attributes['slug'])) {
            return $this->filterIndex?->slug ?? '';
        }
        return $this->attributes['slug'];
    }
}
