<?php

namespace Mtc\MercuryDataModels\Finance\Services;

use Illuminate\Support\Str;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceRequestData;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceResult;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceResultCollection;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceType;
use Mtc\MercuryDataModels\Vehicle;

class IVendiPeterVardy extends IVendi
{
    public function calculate(FinanceRequestData $data): FinanceResultCollection
    {
        return $this->process(
            $this->call($this->mapRequest($data), $data->clientKey),
            $data->finance_option,
            $data->deposit
        );
    }

    protected function process(array $response, $finance_option = 0, float $deposit = 0): FinanceResultCollection
    {
        $finance_product_name_fragment = $this->getProductNameFragmenByFinanceOption($finance_option);

        if (empty($response) || empty($finance_product_name_fragment)) {
            return new FinanceResultCollection();
        }

        $product_quotes = [];

        foreach ($response as $product_group) {
            foreach ($product_group['ProductQuotes'] as $product_quote_data) {
                $finance_type = FinanceType::from($product_quote_data['FacilityType']) ?? null;
                $monthly_payment = $product_quote_data['Figures']['RegularPayment'] ?? null;

                if (
                    !Str::contains($product_quote_data['ProductName'], $finance_product_name_fragment, true)
                    || empty($monthly_payment)
                    || empty($finance_type)
                ) {
                    continue;
                }

                if (!array_key_exists($finance_type->value, $product_quotes)) {
                    $product_quotes[$finance_type->value] = [];
                }

                // get the cheapest quote of each type, ignoring finance option.
                if (
                    empty($product_quotes[$finance_type->value])
                    || $monthly_payment < $product_quotes[$finance_type->value]->monthly_price
                ) {
                    $product_quotes[$finance_type->value] = new FinanceResult(
                        provider: 'ivendi',
                        finance_type: $finance_type,
                        monthly_price: $product_quote_data['Figures']['RegularPayment'] ?? 0,
                        term: $product_quote_data['Figures']['Term'] ?? 0,
                        number_of_payments: $product_quote_data['Figures']['NumberOfRegularPayments'] ?? 0,
                        total_deposit: $product_quote_data['Figures']['TotalDeposit'] ?? 0,
                        apr: $product_quote_data['Figures']['APR'] ?? 0,
                        first_payment: $product_quote_data['Figures']['FirstPayment'] ?? 0,
                        final_payment: $product_quote_data['Figures']['FinalPayment'] ?? 0,
                        interest_rate: $product_quote_data['Figures']['InterestRate'] ?? 0,
                        apply_url: '',
                        product_name: $product_quote_data['ProductName'] ?? '',
                        annual_mileage: $product_quote_data['Figures']['Asset']['AnnualDistanceQuoted'] ?? 0,
                        cash_price: $product_quote_data['Figures']['TotalCashPrice'] ?? 0,
                        dealer_deposit: $product_quote_data['Figures']['Asset']['FinanceDepositAllowance'] ?? 0,
                        customer_deposit: $deposit,
                        payable_amount: $product_quote_data['Figures']['TotalPayable'] ?? 0,
                        option_to_purchase_fee: $product_quote_data['Figures']['OptionToPurchaseFee'] ?? 0,
                        documentation_fee: 0,
                    );
                }
            }
        }

        // We are keying by finance type i.e. HP, PCP etc.
        // The client wants PCP to show first, so we can sort keys in descending order.
        krsort($product_quotes);

        return new FinanceResultCollection($product_quotes);
    }

    public function dealerIdDataField(): string
    {
        return $this->getDealerIdDataField();
    }

    public function dealerSpecificAccountKey(): string
    {
        return $this->getDealerSpecificAccountKey();
    }

    public function isEligibleForCreditRatingFinance(Vehicle $vehicle): bool
    {
        if (strtoupper($vehicle->fuelType?->name) == 'ELECTRIC') {
            return false;
        }

        if (
            $vehicle->getCustom('finance_option') != 1
            && $vehicle->getCustom('finance_option') != 4
        ) {
            return false;
        }

        return $vehicle->dealership->data['finance-risk-based-pricing'] ?? false;
    }

    public function calculateRiskBased(FinanceRequestData $data): FinanceResultCollection
    {
        return $this->processRiskBased(
            $this->call($this->mapCreditRatingRequest($data), $data->clientKey),
            $data->deposit
        );
    }

    public function processRiskBased(array $response, float $deposit): FinanceResultCollection
    {
        if (empty($response)) {
            return new FinanceResultCollection();
        }

        $quotes_by_type = [];

        foreach ($response as $product_group) {
            foreach ($product_group['ProductQuotes'] as $quote) {
                $finance_type = FinanceType::from($quote['FacilityType']) ?? null;

                if (
                    empty($finance_type)
                    || empty($quote['Figures'])
                ) {
                    continue;
                }

                if (!array_key_exists($finance_type->value, $quotes_by_type)) {
                    $quotes_by_type[$finance_type->value] = [];
                }

                // get the cheapest quote of each type, ignoring finance option.
                if (
                    empty($quotes_by_type[$finance_type->value])
                    || $quote['Figures']['RegularPayment'] < $quotes_by_type[$finance_type->value]->monthly_price
                ) {
                    $quotes_by_type[$finance_type->value] = new FinanceResult(
                        provider: 'ivendi',
                        finance_type: $finance_type,
                        monthly_price: $quote['Figures']['RegularPayment'] ?? 0,
                        term: $quote['Figures']['Term'] ?? 0,
                        number_of_payments: $quote['Figures']['NumberOfRegularPayments'] ?? 0,
                        total_deposit: $quote['Figures']['TotalDeposit'] ?? 0,
                        apr: $quote['Figures']['APR'] ?? 0,
                        first_payment: $quote['Figures']['FirstPayment'] ?? 0,
                        final_payment: $quote['Figures']['FinalPayment'] ?? 0,
                        interest_rate: $quote['Figures']['InterestRate'] ?? 0,
                        apply_url: '',
                        product_name: $quote['ProductName'] ?? '',
                        annual_mileage: $quote['Figures']['Asset']['AnnualDistanceQuoted'] ?? 0,
                        cash_price: $quote['Figures']['TotalCashPrice'] ?? 0,
                        dealer_deposit: $quote['Figures']['Asset']['FinanceDepositAllowance'] ?? 0,
                        customer_deposit: $deposit,
                        payable_amount: $quote['Figures']['TotalPayable'] ?? 0,
                        option_to_purchase_fee: $quote['Figures']['OptionToPurchaseFee'] ?? 0,
                        documentation_fee: 0,
                    );
                }
            }
        }

        return new FinanceResultCollection($quotes_by_type);
    }

    /**
     * return the expected product name fragment for a finance option.
     *
     * iVendi should return finance products with names like:
     *      Blackhorse HP B Wee Gem
     *      Blackhorse HP A Standard
     *      Santander PCP E VRM
     *
     * note that we may not have a 1=A, 2=B, 3=C mapping for products.
     * 2 = C 'Best Buy'
     * 3 = B 'Wee Gem'
     *
     * @param int $finance_option
     * @return string
     */
    private function getProductNameFragmenByFinanceOption($finance_option = 0): string
    {
        return match ($finance_option) {
            1 => 'A Standard',
            2 => 'C Best Buy',
            3 => 'B Wee Gem',
            4 => 'D Pre Reg',
            5 => 'E VRM',
            default => ''
        };
    }
}
