<?php

namespace Mtc\MercuryDataModels\Finance\Services;

use Mtc\MercuryDataModels\Finance\Contracts\FinanceRequestData;
use Mtc\MercuryDataModels\Finance\Contracts\FinanceResultCollection;

class CodeWeaversPeterVardy extends CodeWeavers
{
    /**
     * @param FinanceRequestData $data
     * @return FinanceResultCollection
     */
    public function calculate(FinanceRequestData $data): FinanceResultCollection
    {
        return $this->returnCalculationResult($data);
    }

    /**
     * @param FinanceRequestData $data
     * @return FinanceResultCollection
     */
    public function calculateRiskBased(FinanceRequestData $data): FinanceResultCollection
    {
        return $this->returnCalculationResult($data, true);
    }

    private function returnCalculationResult(FinanceRequestData $data, $risk_based = false): FinanceResultCollection
    {
        $finance_types = [
            'HP',
            'PCP',
        ];

        $results = collect($finance_types)->mapWithKeys(function ($finance_type) use ($data, $risk_based) {
            $data->extra['product_type'] = $this->generateProductKey($data, $finance_type);
            $result = $this->runCalculate($this->mapRequest(collect([$data]), $risk_based));

            if (
                $result
                && array_key_exists(0, $result)
                && array_key_exists('Quote', $result[0])
                && array_key_exists('QuoteReference', $result[0]['Quote'])
            ) {
                $result[0]['terms_and_conditions'] = $this->getTermsAndConditions(
                    $result[0]['Quote']['QuoteReference']
                );
            }

            return [
                $finance_type => $this->process($result)->first()
            ];
        });

        return new FinanceResultCollection($results);
    }

    /**
     * Generate the product key required based on the request data.
     *
     * @param FinanceRequestData $data
     * @param string $type
     * @return string
     */
    protected function generateProductKey(FinanceRequestData $data, string $type): string
    {
        // some dealerships require us to use 'HP' and 'PCP' as the keys
        if (stripos($data->clientKey, 'porsche') !== false) {
            return $type;
        }

        if (
            stripos($data->clientKey, 'carz') !== false
            && $data->is_new
            && strtolower($data->make_name) == 'omoda'
        ) {
            return 'OMODA_' . $type;
        }

        if ($data->is_new) {
            return $type;
        }

        $code = $this->getCode($data->finance_option);

        return 'CAT_' . $code . '_' . $type;
    }

    /**
     * return the product code for a finance option.
     *
     * note that we may not have a 1=A, 2=B, 3=C mapping for products.
     * 2 = C 'Best Buy'
     * 3 = B 'Wee Gem'
     *
     * @param int|null $finance_option
     * @return string
     */
    protected function getCode(?int $finance_option): string
    {
        return match ($finance_option) {
            1 => 'A',
            2 => 'C',
            3 => 'B',
            4 => 'D',
            5 => 'E',
            default => '',
        };
    }
}
