<?php

namespace Mtc\MercuryDataModels\Filters;

use App\Facades\Settings;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Str;

class TrimFilter extends IsFilter
{
    /**
     * Apply selections to current filtered object
     *
     * @param Builder $query
     * @param array $selection
     * @return void
     */
    public function applyFilter($query, array $selection = [])
    {
        $query->whereIn('trim', $selection);
    }

    /**
     * Get available results of this filter type
     *
     * @param \Closure $product_filtering
     * @param int $limit
     * @param array $selections
     * @return Collection
     */
    public function getResults(\Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        return $this->getQuery()
            ->distinct()
            ->when(Settings::get('filter-apply-selections-to-results'), fn($query) => $query->where($product_filtering))
            ->select('trim')
            ->get();
    }

    /**
     * Format result for front-end
     *
     * @param Collection $collection
     * @return array
     */
    public function format(Collection $collection): array
    {
        return [
            'title' => $this->title(),
            'ui_component' => $this->uiComponentType(),
            'hide_in_ui' => $this->hideInUi(),
            'has_load_more' => $this->hasLoadMore($collection),
            'show' => $this->visibleByDefault(),
            'min_count_to_show' => $this->minCountToShow(),
            'results' => $collection
                ->filter(fn(Model $model) => !empty(trim($model[$this->getNameAttribute()])))
                ->map(fn(Model $model) => [
                    'id' => Str::slug($model->trim),
                    'name' => $model->trim,
                    'count' => $model->result_count,
                    'slug' => Str::slug($model->trim),
                ]),
        ];
    }

    /**
     * Specify model that drives this filter option.
     * Used to build up filter index.
     *
     * @return string
     */
    public function getModel(): string
    {
        return $this->filter_target_model;
    }

    /**
     * Customer facing name of the filter
     *
     * @return string
     */
    public function title(): string
    {
        return 'Vehicle Trim';
    }

    /**
     * Specify how a slug is formed for this object
     *
     * @param Model $model
     * @return string
     */
    public function modelSlug(Model $model): string
    {
        return $model->trim ?? '';
    }

    /**
     * Specify attribute on object that represents id
     *
     * @return string
     */
    public function getIdAttribute(bool $for_index = false): string
    {
        return 'trim';
    }

    /**
     * Specify attribute on object that represents name
     *
     * @return string
     */
    public function getNameAttribute(bool $for_index = false): string
    {
        return 'trim';
    }

    /**
     * Whether this filter is visible by default
     * @return bool
     */
    public function visibleByDefault(): bool
    {
        return true;
    }

    public function filterType(): string
    {
        return 'trim';
    }
}
