<?php

namespace Mtc\MercuryDataModels\Filters;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Mtc\MercuryDataModels\Contracts\MinMaxFilter;
use Mtc\MercuryDataModels\Vehicle;

class ManufactureYearMaxFilter extends ManufactureYearMinFilter implements MinMaxFilter
{
    private ?int $min_value = null;

    public function applyFilter($query, array $selection = [])
    {
        $query->where('manufacture_year', '<=', $selection);
    }

    public function getResults(\Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        return $this->getData($product_filtering, $limit, $selections)
            ->filter(fn($year) => $this->isInBounds($year))
            ->map(fn ($year) => [
                'id' => $year,
                'name' => $this->getSelectionName($year),
            ]);
    }

    /**
     * Text format of the selection name
     *
     * @param $selection
     * @return string
     */
    public function getSelectionName($selection): string
    {
        return 'Year Up To ' . $selection;
    }

    /**
     * Check if $selection (url slug) matches pattern on this filter
     *
     * @param string $selection
     * @return bool
     */
    public function patternMatched(string $selection): bool
    {
        return preg_match('/year-up-to-[0-9]+/', $selection);
    }

    /**
     * Decode $selection url slug to selection value(s)
     *
     * @param string $selection
     * @return string|array
     */
    public function matchSelections(string $selection)
    {
        preg_match('/year-up-to-([0-9]+)/', $selection, $matches);
        return $matches[1];
    }

    /**
     * Create URL slug for $selection value
     *
     * @param $selection
     * @return string
     */
    public function createSlug($selection): string
    {
        return 'year-up-to-' . $selection;
    }

    protected function isRangeMaximum(): bool
    {
        return true;
    }

    public function filterType(): string
    {
        return 'year_max';
    }

    public function getOtherBound(): string
    {
        return 'manufacture_year_min';
    }

    public function setOtherBoundary(?array $value = null): void
    {
        if (!empty($value)) {
            $this->min_value = $value[0];
        }
    }

    private function isInBounds($value): bool
    {
        if (empty($this->min_value)) {
            return true;
        }
        return $value >= $this->min_value;
    }
}
