<?php

namespace Mtc\MercuryDataModels\Filters;

use Illuminate\Database\Eloquent\Builder;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Illuminate\Support\Facades\Config;
use Illuminate\Support\Facades\Log;
use Mtc\Filter\Contracts\CustomPatternFilter;
use Mtc\MercuryDataModels\Vehicle;

use function collect;

class InsuranceGroupFilter extends IsFilter implements CustomPatternFilter
{
    /**
     * Apply selections to current filtered object
     *
     * @param Builder $query
     * @param array $selection
     * @return void
     */
    public function applyFilter($query, array $selection = [])
    {
        $query->whereHas('attributeValues', fn($query) => $query->where('slug', 'insurance_group')
            ->where('value_integer', '<=', max($selection) ?: PHP_INT_MAX));
    }

    /**
     * Get available results of this filter type
     *
     * @param \Closure $product_filtering
     * @param int $limit
     * @param array $selections
     * @return Collection
     */
    public function getResults(\Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        $ranges = collect(Config::get('automotive.filter-ranges.insurance-groups'));
        return $ranges
            ->map(fn ($range) => [
                'id' => $range,
                'name' => $this->getSelectionName($range)
            ])
            ->values();
    }

    /**
     * Specify model that drives this filter option.
     * Used to build up filter index.
     *
     * @return string
     */
    public function getModel(): string
    {
        return '';
    }

    /**
     * Customer facing name of the filter
     *
     * @return string
     */
    public function title(): string
    {
        return 'Insurance Group';
    }

    /**
     * Specify how a slug is formed for this object
     *
     * @param Model $model
     * @return string
     */
    public function modelSlug(Model $model): string
    {
        return '';
    }

    /**
     * Format result for front-end
     *
     * @param Collection $collection
     * @return array
     */
    public function format(Collection $collection): array
    {
        return [
            'title' => $this->title(),
            'ui_component' => $this->uiComponentType(),
            'hide_in_ui' => $this->hideInUi(),
            'has_load_more' => $this->hasLoadMore($collection),
            'results' => $collection,
            'show' => $this->visibleByDefault(),
            'min_count_to_show' => $this->minCountToShow(),
        ];
    }

    /**
     * Check if $selection (url slug) matches pattern on this filter
     *
     * @param string $selection
     * @return bool
     */
    public function patternMatched(string $selection): bool
    {
        return preg_match('/insurance-up-to-[0-9]+/', $selection);
    }

    /**
     * Decode $selection url slug to selection value(s)
     *
     * @param string $selection
     * @return string|array
     */
    public function matchSelections(string $selection)
    {
        preg_match('/insurance-up-to-([0-9]+)/', $selection, $matches);
        return $matches[1];
    }

    /**
     * Create URL slug for $selection value
     *
     * @param $selection
     * @return string
     */
    public function createSlug($selection): string
    {
        return 'insurance-up-to-' . $selection;
    }

    /**
     * Text format of the selection name
     *
     * @param $selection
     * @return string
     */
    public function getSelectionName($selection): string
    {
        return 'Insurance group ' . (int)$selection . ' and under';
    }
}
