<?php

namespace Mtc\MercuryDataModels\Filters;

use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Collection;
use Mtc\MercuryDataModels\Franchise;

class FranchiseFilter extends IsFilter
{
    /**
     * Apply selections to current filtered object
     *
     * @param Builder $query
     * @param array $selection
     * @return void
     */
    public function applyFilter($query, array $selection = [])
    {
        $query->whereHas('dealership.franchise', fn ($sub_query) => $sub_query->whereIn('slug', $selection));
    }

    /**
     * Get available results of this filter type
     *
     * @param \Closure $product_filtering
     * @param int $limit
     * @param array $selections
     * @return Collection
     */
    public function getResults(\Closure $product_filtering, int $limit, array $selections = []): Collection
    {
        return Franchise::query()
            ->distinct()
            ->get();
    }

    public function format(Collection $collection): array
    {
        return [
            'title' => $this->title(),
            'ui_component' => $this->uiComponentType(),
            'has_load_more' => $this->hasLoadMore($collection),
            'show' => $this->visibleByDefault(),
            'hide_in_ui' => true,
            'min_count_to_show' => $this->minCountToShow(),
            'results' => $collection
                ->filter(fn(Model $model) => !empty(trim($model[$this->getNameAttribute()])))
                ->map(fn(Model $model) => [
                    'id' => $model->slug ?: $model[$this->getIdAttribute()],
                    'name' => $model[$this->getNameAttribute()],
                    'count' => $model->result_count,
                    'slug' => $model->slug
                ]),
        ];
    }

    /**
     * Specify model that drives this filter option.
     * Used to build up filter index.
     *
     * @return string
     */
    public function getModel(): string
    {
        return Franchise::class;
    }

    /**
     * Customer facing name of the filter
     *
     * @return string
     */
    public function title(): string
    {
        return 'Franchise';
    }

    /**
     * Specify how a slug is formed for this object
     *
     * @param Model $model
     * @return string
     */
    public function modelSlug(Model $model): string
    {
        return $model->slug;
    }
}
